/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2014-2016 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat;

import org.silvestrislab.cyclotis.omegat.tm.PropertiesStore;
import org.silvestrislab.cyclotis.omegat.proj.ctx.*;

import org.omegat.core.data.PrepareTMXEntry;
import org.omegat.util.TMXProp;

import java.sql.*;
import java.util.List;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.*;

public abstract class PostgresqlCyclotisTMX<T extends PrepareTMXEntry> extends PostgresqlCyclotis<T> {
	
	public abstract static class PROPS_COLS_TYPE {
		public abstract Object write (List<TMXProp> properties);
		public abstract List<TMXProp> read (Object properties);
		public String declaredType() { return ""; }
		
		public String toString() { return getClass().getName().substring(0, getClass().getName().indexOf("Cols")); }
	}
	
	public static class TextColsType extends PROPS_COLS_TYPE {
		public String write (List<TMXProp> properties) {
			StringBuffer buffer = new StringBuffer("TABLE:");
			for (TMXProp prop: properties) buffer.append(prop.getType()).append("=").append(prop.getValue()).append("&");
			return buffer.toString();
		}
		public List<TMXProp> read (Object properties) {
			List<TMXProp> props = new ArrayList<TMXProp> ();
			String propStr = properties.toString(); propStr = propStr.replace("TABLE:","");
			for (String prop0: propStr.split("&")) {
				String[] keyVal = prop0.split("=");
				props.add (new TMXProp (keyVal[0], keyVal[1]));
			}
			return props;
		}
	}
	
	public static class JsonColsType extends PROPS_COLS_TYPE {
		public String write (List<TMXProp> properties) {
			StringBuffer buffer = new StringBuffer("{ ");
			for (TMXProp prop: properties) buffer.append("\"" + prop.getType() + "\"").append(" : ").append("\"" + prop.getValue() + "\"").append(",");
			buffer.replace (buffer.length() - 1, buffer.length(), "}"); 
			return buffer.toString();		
		}
			
		private final static Pattern JSON_PROP = Pattern.compile("\\\"(.+?)\\\"\\s*:\\s*\\\"(.+?)\\\"");
			
		public List<TMXProp> read (Object properties) {
			List<TMXProp> props = new ArrayList<TMXProp> ();
			Matcher matcher = JSON_PROP.matcher(properties.toString());
			while (matcher.find()) props.add (new TMXProp (matcher.group(1), matcher.group(2)));
			return props;
		}	

		public String declaredType() { return "json"; }
			
	}
	
	public static class HstoreColsType extends PROPS_COLS_TYPE {
		public Map<String,String> write (List<TMXProp> properties) {
			HashMap<String,String> map = new HashMap<String,String> ();
			for (TMXProp prop: properties) map.put (prop.getType(), prop.getValue());
			return map;
		}
		
		private final static Pattern HSTORE_PROP = Pattern.compile("\\\"(.+?)\\\"\\s*=>\\s*\\\"(.+?)\\\"");
		
		public List<TMXProp> read (Object properties) {
			List<TMXProp> props = new ArrayList<TMXProp> ();
			if (properties instanceof String) {
				Matcher matcher = HSTORE_PROP.matcher(properties.toString());
				while (matcher.find()) props.add (new TMXProp (matcher.group(1), matcher.group(2)));
			}
			if (properties instanceof Map) {
				for (Map.Entry<String,String> hme: ((Map<String,String>) properties).entrySet())
					props.add (new TMXProp (hme.getKey(), hme.getValue()));
			}
			return props;
		}
		
	}
	
	public static class CompositeColsType extends PROPS_COLS_TYPE {
		private String typeName;
		private String[] cols;
 		
		public CompositeColsType(String name,String[] cols) { this.typeName = name; this.cols = cols; }
		
		public String declaredType() { return typeName; }
	
		public String write (List<TMXProp> properties) {
			StringBuffer buffer = new StringBuffer("(");
			for (String col0: cols) {
				String value = "";
				for (TMXProp prop: properties)
					if (prop.getType().equalsIgnoreCase(col0))
						value = prop.getValue();
				buffer.append(value).append(",");
			}
			buffer.setLength(buffer.length() - 1);
			System.out.println("Writing props = " + buffer);
			return buffer.append(")").toString();
		}
				
		public List<TMXProp> read (Object properties) {
			Object[] attributes = null;
			if (properties instanceof java.sql.Struct) 
				try {
					attributes = ((java.sql.Struct) properties).getAttributes();
				} catch (Exception e) {
					e.printStackTrace();
				}
			else { // Try as a string
				String propsString = properties.toString();
				propsString = propsString.substring(1);
				propsString = propsString.substring(0, propsString.indexOf(")"));
				attributes = propsString.split(",");
			}
			List<TMXProp> props = new ArrayList<TMXProp> ();
			if (attributes != null)
				for (int i = 0; i < Math.min(attributes.length, cols.length); i++)
					props.add (new TMXProp (cols[i], attributes[i].toString()));
			return props;
		}
		
		public String toString() { 
			StringBuffer buf = new StringBuffer("Composite type ").append(typeName).append("(");
			for (String col: cols) buf.append(col).append(",");
			return buf.toString();
		}
	}

	
	protected PROPS_COLS_TYPE col_props;
	protected final PropertiesStore store;
	protected ContextMode contextMode;
	
	protected PostgresqlCyclotisTMX (java.util.Properties propList) throws java.sql.SQLException, ClassNotFoundException { 
		super(propList); 
		this.store = new PropertiesStore(this, propList);
	}
	
	protected PostgresqlCyclotisTMX (PostgresqlCyclotisTMX ori) {
		super(ori);
		this.col_props = ori.col_props; this.store = ori.store; this.contextMode = ori.contextMode;
	}

	protected void registerColumn(ResultSet set, java.util.Properties propList) throws SQLException {
		if (set.getString("column_name").equalsIgnoreCase("PROPS")) {
			String name = set.getString("udt_name").toUpperCase();
			if (name.startsWith("VARCHAR") || name.equals("TEXT")) col_props = new TextColsType();
			else if (name.equals("HSTORE")) col_props = new HstoreColsType();
			else if (name.equals("JSON")) col_props = new JsonColsType();
			else {
				PreparedStatement pColumnInfo = connection.prepareStatement ("select attribute_name from INFORMATION_SCHEMA.ATTRIBUTES where udt_name = ? and udt_schema = ? and udt_catalog = current_catalog order by ordinal_position");
				pColumnInfo.setString(1, name.toLowerCase());
				pColumnInfo.setString(2, set.getString("udt_schema"));
				ResultSet set1 = pColumnInfo.executeQuery(); List<String> attNames = new ArrayList<String>();
				while (set1.next()) attNames.add (set1.getString("attribute_name"));
				col_props = new CompositeColsType(name, attNames.toArray(new String[0]));
			}
			logMessage("struct", "registerColumn: " + this + " .col_props = " + col_props);
		}
		if (set.getString("column_name").equalsIgnoreCase("CONTEXT")) {
			String type = set.getString("udt_name").toUpperCase();
			if (type.startsWith("INT")) contextMode = IntContextMode.forProperties(propList); 
			if (type.startsWith("VARCHAR")) contextMode = StringContextMode.forProperties(propList, set.getInt("character_maximum_length")); 
			if (type.startsWith("TEXT")) contextMode = StringContextMode.forProperties(propList, Integer.MAX_VALUE);
			if (type.startsWith("BPCHAR")) contextMode = StringContextMode.forProperties(propList, -set.getInt("character_maximum_length")); 
			logMessage("struct", "Context mode is " + contextMode + " (type was " + type +")");
		}
		super.registerColumn(set, propList);		
	}
	
	@Override
	protected String getInsertFieldType (String name) {
		if (name.equalsIgnoreCase("props")) return col_props.declaredType();
		return ""; 
	}
	
	/* ------------------ IWritableExternalMemory ---------------*/	
	
	public void registerTranslation(T entryContents) throws Exception {
		logMessage("update", "Writing entry '" + entryContents.source + "'");
		try {
			pInsert.setString (1, reformatText(entryContents.source,false)); 
			pInsert.setString (2, reformatText(entryContents.translation,false));
			int res = pInsert.executeUpdate();
			logEntry("search-results", entryContents, false);
			logMessage("update", "entry '" + entryContents.source + "' updated :" + res);
		} catch (Exception e) {
			logException(e); throw e;
		}
	}
		
	/* ------------  Logging -------------- */
	
	public StringBuffer entryToLogString (T entry, boolean detailed, boolean digest) {
		StringBuffer buf;
		if (detailed)
			buf = new StringBuffer("\t<TMX Entry author='" + entry.creator 
				+ "' creation-date='" + LONG_LOG_DATE_FORMAT.format(new java.util.Date(entry.creationDate)) + "'"
				+ " changer = '" + entry.changer + "' change-date='" + LONG_LOG_DATE_FORMAT.format(new java.util.Date(entry.changeDate)) + ">\n");
		else
			buf = new StringBuffer("\t<TMX Entry>\n");			
		if (! digest) {
			buf.append ("\t\t<source>" + entry.source + "</source>\n");		
			buf.append ("\t\t<target>" + entry.translation + "</target>\n");
		} else {
			buf.append ("\t\t<source digest='" + digest(entry.source) + "'>" + entry.source + "</source>\n");		
			buf.append ("\t\t<target digest='" + digest(entry.translation) + "'>" + entry.translation + "</target>\n");			
		}
		buf.append ("\t</TMX Entry>");
		return buf;
	}

	
}