/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2014 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat.proj.ctx;

import org.omegat.core.data.SourceTextEntry;
import org.omegat.core.matching.external.ProjectMemory.ContextTMXEntry;

import org.silvestrislab.cyclotis.omegat.Cyclotis;

import java.sql.*;

public abstract class StringContextMode extends ContextMode<String> {
	
	public final void fillEntry (Cyclotis origin, ContextTMXEntry entry, ResultSet data) throws SQLException {
		this.fillEntry (origin, entry, data.getString("CONTEXT"));
	}

	public void setStatementValue (PreparedStatement stmt, int pos, SourceTextEntry key, boolean isDefault) throws SQLException {
		stmt.setString (pos, this.toDatabase (key, isDefault));
	}

	public StringContextMode nullWrapper(final int sqlType) {
		return new StringContextMode () {
			protected String convert (String val) { return StringContextMode.this.convert(val); }
			public String defaultValueForSource (String sourceText) { return null; }
			public String nonDefaultValueForEntry (SourceTextEntry key) { return StringContextMode.this.nonDefaultValueForEntry(key); }
			public boolean isDefault (String value) { return value == null; }
			public boolean isDefault (SourceTextEntry entry, String value) { return value == null; }
			public boolean isCorrectKey (SourceTextEntry key, String value) { return StringContextMode.this.isCorrectKey(key,value); }
			public void setStatementValue (PreparedStatement stmt, int pos, SourceTextEntry entry, boolean isDefault) throws SQLException {
				if (isDefault) stmt.setNull (pos, sqlType);
				else super.setStatementValue (stmt, pos,entry, false);
			}
			public String toString() { return "null or " + StringContextMode.this.toString(); }
		};
	}
	
	protected abstract String convert(String val);
	
	public StringContextMode constWrapper(final String constVal, final boolean convert) {
		return new StringContextMode () {
			private String constConverted = convert ? StringContextMode.this.convert (constVal) : constVal;		
			protected String convert (String val) { return StringContextMode.this.convert(val); }
			
			public String defaultValueForSource (String sourceText) { return constConverted; }
			public String nonDefaultValueForEntry (SourceTextEntry key) { return StringContextMode.this.nonDefaultValueForEntry(key); }
			public boolean isDefault (String value) { return value == null || value.equals(constConverted); }
			public boolean isDefault (SourceTextEntry entry, String value) { return value == null || value.equals(constConverted); }
			public boolean isCorrectKey (SourceTextEntry entry, String value) { return StringContextMode.this.isCorrectKey(entry,value); }
			public void setStatementValue (PreparedStatement stmt, int pos, SourceTextEntry entry, boolean isDefault) throws SQLException {
				if (isDefault) stmt.setString(pos, constConverted);
				else stmt.setString(pos, this.nonDefaultValueForEntry(entry));
			}
			public String toString() { return "default(" + constVal + ") or " + StringContextMode.this.toString(); }
		};
	}	
	
	public static final StringContextMode DEFAULT = new StringWrapperContextMode(new HashContextMode());
	
	public static StringContextMode forProperties (java.util.Properties propList, int col_context_type) {
		ContextMode<?> mode = null;
		try {
			mode = ContextMode.forProperties(propList);
		} catch (Exception e1) {
			mode = DEFAULT;
		}
		if (mode == null)
			try {
				switch (col_context_type) {
				case +32: case -32:
					mode = new DigestStringContextMode("MD5");
					break;
				case +40: case -40:
					mode = new DigestStringContextMode("SHA1");
					break;
				default:
					// We can use whatever we want but we use the smallest possible value
					mode = DEFAULT;
				}
			} catch (Exception eTyp) {
				mode = DEFAULT;
			}		
		try {
			StringContextMode sMode = (StringContextMode) mode;
			String contextDefault = propList.getProperty("context.default");
			if (contextDefault != null) {
				if (contextDefault.equalsIgnoreCase("null")) 
					return sMode.nullWrapper(col_context_type > 0 ? java.sql.Types.VARCHAR : java.sql.Types.CHAR);
				else if (contextDefault.toLowerCase().startsWith("const")) {
					contextDefault = contextDefault.substring(6);
					if (col_context_type < 0) { // CHAR(n) : we must be sure to use correct size
						while (contextDefault.length() < (-col_context_type)) contextDefault += " "; // shorter, so add spaces
						contextDefault = contextDefault.substring(0, -col_context_type); // longer, so truncate
					}
					return sMode.constWrapper(contextDefault, false);
				}
				else if (contextDefault.toLowerCase().startsWith("convert")) 
					return sMode.constWrapper(contextDefault.substring(8), true);
				else if (contextDefault.toLowerCase().startsWith("empty")) // empty string (""), with or without conversion
					return sMode.constWrapper("", contextDefault.toLowerCase().contains("conv"));
			}
			else return sMode;
		} catch (Exception sNo) {
			try {
				final IntContextMode iMode = (IntContextMode) mode;
				return new StringWrapperContextMode(iMode);
			} catch (Exception e) {
				return DEFAULT;
			}
		}
		return DEFAULT;
	}
	
}

final class StringWrapperContextMode extends StringContextMode {
	private IntContextMode iMode;
	
	public StringWrapperContextMode(IntContextMode iMode) {
		this.iMode = iMode;
	}

	public String defaultValueForSource (String sourceText) {
		return iMode.defaultValueForSource(sourceText).toString();
	}
	
	public String nonDefaultValueForEntry (SourceTextEntry key) {
		return iMode.nonDefaultValueForEntry(key).toString();
	}
	
	public boolean isDefault (String value) {
		try {
			return iMode.isDefault (Integer.parseInt (value));
		} catch (Exception e) {
			return false;
		}
	}
	
	public boolean isDefault (SourceTextEntry key, String value) {
		try {
			return iMode.isDefault (key, Integer.parseInt (value));
		} catch (Exception e) {
			return false;
		}
	}
	
	public boolean isCorrectKey (SourceTextEntry key, String value) {
		try {
			return iMode.isCorrectKey (key, Integer.parseInt (value));
		} catch (Exception e) {
			return false;
		}
	}
	
	public String convert(String val) {
		return val;
	}
	
	public String toString () {
		return "toString(" + iMode + ")";
	}

	public boolean isUnique() { return iMode.isUnique(); } 
	
}