/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2014-2016 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat.tm;

import org.silvestrislab.cyclotis.omegat.Cyclotis;

import org.omegat.core.data.PrepareTMXEntry;
import org.omegat.core.data.SourceTextEntry;
import org.omegat.util.TMXProp;
import org.omegat.util.Language;

import java.util.regex.*;
import java.util.*;
import java.sql.*;

public class PropertiesStore  {

	private Properties propList;
	private Cyclotis parent;
	
	public PropertiesStore (Cyclotis parent, Properties propList) {
		this.parent = parent;
		this.propList = propList;
	}
	
	public static final String PROP_ENTRY_NUM = "entry.num";
	public static final String PROP_ENTRY_FILE = "entry.file";
	public static final String PROP_ENTRY_ID = "entry.id";
	public static final String PROP_ENTRY_PATH = "entry.path";
	public static final String PROP_TABLE_NAME = "db.table.name";

	public String sysVariable(String name, PrepareTMXEntry entryContents, SourceTextEntry entrySource) {
		if (PROP_ENTRY_NUM.equals(name)) return Integer.toString(entrySource.entryNum());
		if (PROP_ENTRY_FILE.equals(name)) return entrySource.getKey().file;
		if (PROP_ENTRY_ID.equals(name)) return entrySource.getKey().id == null ? "null" : entrySource.getKey().id;
		if (PROP_ENTRY_PATH.equals(name))return entrySource.getKey().path == null ? "null" : entrySource.getKey().path;
		if (name.startsWith("tu.")) {
			if ("tu.author".equals(name)) 
				if (entryContents.changer != null) return entryContents.changer; else return entryContents.creator;
			if ("tu.creator".equals(name)) return entryContents.creator;
			if ("tu.changer".equals(name)) return entryContents.changer;
			if ("tu.creationDate".equals(name)) return Long.toString(entryContents.creationDate);
			if ("tu.changeDate".equals(name)) return Long.toString(entryContents.changeDate);
			if ("tu.date".equals(name))
				if (entryContents.changeDate > 0) return Long.toString(entryContents.changeDate);
				else return Long.toString(entryContents.creationDate);
		}
		return "";
	}
	
	private static final Pattern patternPropertyIf = Pattern.compile("\\.((\\d+)_)?if\\Z");
	private static final Pattern patternPropertyVal = Pattern.compile("\\.((\\d+)_)?value\\Z");
	
	public List<TMXProp> toDatabase (PrepareTMXEntry entryContents, SourceTextEntry entrySource) {
		List<TMXProp> result = new ArrayList<TMXProp>();
		for (String name: propList.stringPropertyNames())
			if (name.startsWith("props.toDatabase")) {
				String subname = name.substring(17), val = propList.getProperty(name);
				Matcher matchIf = patternPropertyIf.matcher(subname);
				if (matchIf.find()) {
					subname = subname.substring(0, subname.length() - matchIf.group(0).length()); String condString = val; 
					if (matchIf.group(2).length() > 0) val = propList.getProperty("props.toDatabase." + subname + "." + matchIf.group(2) + "_value");
					else val = propList.getProperty("props.toDatabase." + subname + ".value");
					parent.logMessage ("props2db", "Conditional property: "  + condString);
					SaveConditions conditions = SaveConditions.forProperty(condString);
					if (conditions.mustWrite(this.parent, entryContents, entrySource)) {
						parent.logMessage ("props2db", "Store property: "  + subname + " from " + val);
						val = interpolation (val, result, entryContents, entrySource);
						parent.logMessage ("props2db", "Store property: "  + subname + " = " + val);
						result.add (new TMXProp (subname, val));
					}
					else parent.logMessage ("props2db", "Condition is false");
				} 
				else if (patternPropertyVal.matcher(subname).find()) { } // associated with "if" : do nothing
				else {
					parent.logMessage ("props2db", "Store property: "  + subname + " from " + val);
					val = interpolation (val, result, entryContents, entrySource);
					parent.logMessage ("props2db", "Store property: "  + subname + " = " + val);
					result.add (new TMXProp (subname, val));
				}
			}
		return result;
	}
		
	public static final String expr_split = "\\|([^\\|]+)\\|(\\d+)";
	public static final String expr_substring = "\\:(\\d+)(?:,(\\d+))?";
	public static final String expr_format = "\\!\\%([0-9\\.]*)([dfx])";
	
    public static final Pattern patternSingleProperty = Pattern.compile("\\@\\{([\\w\\.]+?)(?:" + expr_split + ")?(?:" + expr_substring + ")?(?:" + expr_format + ")?\\}");
    public static final Pattern patternSysVariable = Pattern.compile("\\$\\{([\\w\\.]+?)(?:" + expr_split + ")?(?:" + expr_substring + ")?(?:" + expr_format + ")?\\}");
	
	/** Replaces ${...} or @{...} in val by the corresponding values **/
	private String interpolation (String val, List<TMXProp> extractedList) {
		Matcher matcher;
		while ((matcher = patternSingleProperty.matcher(val)).find()) {
			String propName = matcher.group(1); String result = null;
			for (TMXProp prop: extractedList)
				if (prop.getType().equalsIgnoreCase(propName)) result = prop.getValue();
			result = findPart (result, matcher);
			val = val.replace (matcher.group(), result == null ? "" : result);
		}
		return val;
	}

	/** Replaces ${...} or @{...} in val by the corresponding values **/
	private String interpolation (String val, List<TMXProp> extractedList, PrepareTMXEntry entryContents, SourceTextEntry entrySource) {
		Matcher matcher;
		val = val.replace ("${db.table.name", "@{db.table.name"); // indeed it is in the extracted list
		while ((matcher = patternSysVariable.matcher(val)).find()) {
			String propName = matcher.group(1); String result = sysVariable (propName, entryContents, entrySource);
			result = findPart (result, matcher);
			val = val.replace (matcher.group(), result == null ? "" : result);
		}
		// Substitute properties after sysvariables, since they could contain them.
		val = interpolation (val, extractedList);
		return val;
	}
	
	/** If the matcher contains expr_split or expr_substring, returns the corresponding subpart. Else return the result. **/
	private String findPart (String result, Matcher matcher) {
		if (matcher.group(2) != null) 
			try {
				String expr = "[" + matcher.group(2) + "]";
				int pos = Integer.parseInt (matcher.group(3));
				String[] split = result.split (expr);
				result = split[pos];
			} catch (Exception e) {
			
			}
		if (matcher.group(4) != null) 
			try {
				int start = Integer.parseInt (matcher.group(4));
				try {
					int end = Integer.parseInt (matcher.group(5));
					result = result.substring (start, end);
				} catch (Exception e) {
					result = result.substring (start);
				}
			} catch (Exception e) {
			
			}
		if (matcher.group(7) != null) 
			try {
				System.out.println("Format " +"%" + matcher.group(6) + matcher.group(7) + " of "+ result);
				char type = matcher.group(7).charAt(0);
				switch (type) {
				case 'd': {
					int i = Integer.parseInt(result);
					System.out.println("result is int");
					result = String.format("%" + matcher.group(6) + matcher.group(7), i);
					break;
				}
				case 'f': {
					float i = Float.parseFloat(result);
					result = String.format("%" + matcher.group(6) + matcher.group(7), i);
					break;
				}
				case 'x': 
					result = String.format("%" + matcher.group(6) + matcher.group(7), result);
					break;
				}
			} catch (Exception e) {
			
			}
			
		return result;
	}
	
	public List<TMXProp> fromDatabase (List<TMXProp> extractedList, PrepareTMXEntry entryContents) {
		List<TMXProp> newList = new ArrayList<TMXProp>();
		if (extractedList == null) extractedList = Collections.<TMXProp>emptyList();
		StringBuffer extractedListBuf = new StringBuffer(); 
		for (TMXProp prop: extractedList) extractedListBuf.append(prop.getType()).append("=").append(prop.getValue()).append(",");
		parent.logMessage ("props2ui", "DB contents: "  + extractedListBuf);		
		for (String name: propList.stringPropertyNames())
			if (name.startsWith("props.display")) {				
				String subname = name.substring(14), val = propList.getProperty(name);
				Matcher matchIf = patternPropertyIf.matcher(subname);
				if (matchIf.find()) {
					subname = subname.substring(0, subname.length() - matchIf.group(0).length()); String condString = val; 
					if (matchIf.group(2).length() > 0) val = propList.getProperty("props.display." + subname + "." + matchIf.group(2) + "_value");
					else val = propList.getProperty("props.display." + subname + ".value");
					parent.logMessage ("props2ui", "Conditional property: "  + condString);
					SaveConditions conditions = SaveConditions.forProperty(condString);
					if (conditions.mustWrite(this.parent, entryContents, null)) {
						parent.logMessage ("props2ui", "Get property: "  + subname + " from " + val);
						val = interpolation (val, extractedList, entryContents, null);
						parent.logMessage ("props2ui", "Get property: "  + subname + " = " + val);
						newList.add (new TMXProp (subname, val));
					}
					else parent.logMessage ("props2ui", "Condition is false");
				}				
				else if (patternPropertyVal.matcher(subname).find()) { } // associated with "if" : do nothing
				else {
					parent.logMessage ("props2ui", "Get property: "  + subname + " from " + val);
					val = interpolation (val, extractedList, entryContents, null);
					parent.logMessage ("props2ui", "Get property: "  + subname + " = " + val);
					newList.add (new TMXProp (subname, val));
				}
			}
		if (propList.getProperty("props.exclude") != null) {
			Set<String> exclude = new HashSet<String>();
			for (String item: propList.getProperty("props.exclude").split(",")) exclude.add (item);
			for (TMXProp prop: extractedList) if (! exclude.contains(prop.getType())) newList.add(prop);
		} else 
			newList.addAll (extractedList);
		return newList;
	}
	
}