/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2014 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat.tm;

import org.omegat.core.data.PrepareTMXEntry;
import org.omegat.core.data.SourceTextEntry;
import org.omegat.core.Core;
import org.omegat.util.*;

import org.silvestrislab.cyclotis.omegat.Cyclotis;

import java.util.regex.*;
import java.util.*;
import java.sql.*;
import javax.script.*;

/**
 * Indicates in which case entry should be written in the database
 **/
public abstract class SaveConditions  {

	/**
	 * This is the method to be implemented to define condition
	 * @param parent	The cyclotis instance. Used for logging
	 * @param entryContents		The translation unit
	 * @param entrySource		The entry in source file
	 **/
	public abstract boolean mustWrite(Cyclotis parent, PrepareTMXEntry entryContents, SourceTextEntry entrySource);
	
	private static final ScriptEngineManager manager = new ScriptEngineManager(forProperty("true").getClass().getClassLoader());
	private static final ScriptEngine scriptEngine = manager.getEngineByName("javascript");
	private static int count = 0;
	
	public static SaveConditions forProperty (final String property) {
		if (property.equalsIgnoreCase("nonfuzzy"))
			return new SaveConditions() {
				public boolean mustWrite(Cyclotis parent, PrepareTMXEntry entryContents, SourceTextEntry entrySource) {  
					return ! (entryContents.translation.startsWith("[fuzzy]")
						||	entryContents.translation.startsWith(OStrings.getString("WF_DEFAULT_PREFIX"))
					);
				}
				public String toString() { return "Non-fuzzy"; }
			};
		if (property.equalsIgnoreCase("nonsrc"))
			return new SaveConditions() {
				public boolean mustWrite(Cyclotis parent, PrepareTMXEntry entryContents, SourceTextEntry entrySource) {  
					return ! (entryContents.translation.equals(entryContents.source));
				}
				public String toString() { return "Non-fuzzy"; }
			};
			
		if (property.startsWith("tra-not:"))
			return new SaveConditions() {
				private final String marker = property.substring(8);
				
				public boolean mustWrite(Cyclotis parent, PrepareTMXEntry entryContents, SourceTextEntry entrySource) {  
					return ! (entryContents.translation.contains(marker));
				}
				public String toString() { return "Tra-Marker:" + marker; }
			};
		if (property.startsWith("note-not:"))
			return new SaveConditions() {
				private final String marker = property.substring(9);
				
				public boolean mustWrite(Cyclotis parent, PrepareTMXEntry entryContents, SourceTextEntry entrySource) {  
					return ! (entryContents.note.contains(marker));
				}
				public String toString() { return "Note-Marker:" + marker; }
			};
			
		
		if (property.toLowerCase().startsWith("js:")) 
			return new SaveConditions() {
				String expression = property.substring(3);				
				int id = count; 
				{
					try {
						scriptEngine.eval("function mustWrite" + count + "(project,contents,source) { return " + expression + "; }");				
						count++; 
					} catch (Exception e) {
						e.printStackTrace();
					}
				}
				
				public boolean mustWrite(Cyclotis parent, PrepareTMXEntry entryContents, SourceTextEntry entrySource) { 
					try {
						Invocable inv = (Invocable) scriptEngine;
						Object eval = inv.invokeFunction("mustWrite" + id, Core.getProject(), entryContents, entrySource);
						parent.logMessage ("javascript", expression + " evaluated as " + eval);
						if (eval == null) return false;
						if (eval.toString().equalsIgnoreCase("false")) return false;
						if (eval.toString().equalsIgnoreCase("0")) return false;
					} catch (Exception e) {
						parent.logMessage ("javascript", expression + " not evaluated : " + e);
						return false;
					}
					return true;
				}

				public String toString() { return "javascript:" + expression; }
			};
	
		// Default
		return new SaveConditions() {
			boolean constValue = property.equalsIgnoreCase("true");
			
			public boolean mustWrite(Cyclotis parent, PrepareTMXEntry entryContents, SourceTextEntry entrySource) { return constValue; }
			public String toString() { return "Always " + constValue; }
		};
	}
	
}