/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2013-2016 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat;

import org.omegat.core.Core;
import org.omegat.core.data.TMXEntry;

import java.text.SimpleDateFormat;
import java.util.Properties;
import java.io.*;
import javax.script.*;
import java.util.regex.*;

public abstract class Cyclotis<T> {

	protected static final SimpleDateFormat SHORT_LOG_DATE_FORMAT = new SimpleDateFormat ("HH:mm:ss.SSS"), LONG_LOG_DATE_FORMAT = new SimpleDateFormat ("MM-dd HH:mm:ss");

	private final static java.util.HashMap<String, FileOutputStream> LOG_STREAMS = new java.util.HashMap<String, FileOutputStream>();
	
	protected String name = "", providerName = "Silvestris Cyclotis";
	private PrintStream logDest = System.err;
	private String logList;
	private ScriptEngine scriptEngine = null;
	
	public Cyclotis (Properties propList) {
		if (propList.getProperty("provider.name") != null) this.providerName = propList.getProperty("provider.name");
		this.name = propList.getProperty ("name");
		if (propList.getProperty("log.file") != null)
			try {
				String fileName = propList.getProperty("log.file");
				if (! (fileName.startsWith("/"))) 
					fileName = Core.getProject().getProjectProperties().getProjectRoot() + File.separator + fileName;
				FileOutputStream fos = LOG_STREAMS.get(fileName);
				if (fos != null) { logDest = new PrintStream (fos); logMessage("", "Shared log for " + providerName + "/" + name); }
				else { 	// new stream
					boolean exist = new File(fileName).exists();
					fos = new FileOutputStream (fileName, true);
					logDest = new PrintStream (fos);
					if (exist) logDest.println("\n\n"); // Leave some space
				}			
			} catch (Exception e) {
				e.printStackTrace();
			}
		if ((propList.getProperty("text.get.script") != null) || (propList.getProperty("text.put.script") != null)) {
			ScriptEngineManager manager = new ScriptEngineManager(Cyclotis.class.getClassLoader());
			this.scriptEngine = manager.getEngineByName("javascript");
			String script = propList.getProperty("text.get.script");
			if (script != null) try { scriptEngine.eval("function pre(text) { " + script + " }"); } catch (Exception e) { e.printStackTrace(); }
			else try { scriptEngine.eval("function pre(text) { return text; }"); } catch (Exception e) { e.printStackTrace(); }
			script = propList.getProperty("text.put.script");
			if (script != null) try { scriptEngine.eval("function post(text) { " + script + " }"); } catch (Exception e) { e.printStackTrace(); }
			else try { scriptEngine.eval("function post(text) { return text; }"); } catch (Exception e) { e.printStackTrace(); }
		}
		this.logList = propList.getProperty("log.list"); if (logList == null) logList = "";
		if (logDest != System.err) {
			String buildDate = "unknown date";
			try {
				String rn = Cyclotis.class.getName().replace('.', '/') + ".class";
				java.net.JarURLConnection conn = (java.net.JarURLConnection) Cyclotis.class.getClassLoader().getResource(rn).openConnection();
				buildDate = LONG_LOG_DATE_FORMAT.format(new java.util.Date(conn.getJarFile().getEntry("META-INF/MANIFEST.MF").getTime()));
			} catch (Exception e) {
				e.printStackTrace();
			}
			logDest.println (LONG_LOG_DATE_FORMAT.format(new java.util.Date()) 
				+ " - Starting logging "
				+ " Cyclotis (" + buildDate + ")"
				+ " for " + providerName + " / " + name);
		}
		if ("hash".equalsIgnoreCase(propList.getProperty("log.digest"))) 
			this.digestAlgo = new org.silvestrislab.cyclotis.omegat.proj.ctx.HashContextMode();
		else if (propList.getProperty("log.digest") != null)
			if (propList.getProperty("log.digest").toLowerCase().startsWith("digest "))
				try { this.digestAlgo = new org.silvestrislab.cyclotis.omegat.proj.ctx.DigestStringContextMode(propList.getProperty("log.digest").substring(7)); }
				catch (Exception e) {  logMessage("", "Cannot create digest algorithm"); logException(e); }
		try { this.limitReduce = Integer.parseInt( propList.getProperty("limit.reduceWords") ); logMessage("config", "Reducing words after " + limitReduce); } catch (Exception e) {}
	}
	
	protected Cyclotis(Cyclotis ori) {
		this.name = ori.name; this.providerName = ori.providerName;
		this.logDest = ori.logDest; this.logList = ori.logList; this.digestAlgo = ori.digestAlgo;
		this.scriptEngine = ori.scriptEngine; this.limitReduce = ori.limitReduce;
	}	
	
	/* ------------------  Transformation of source or target text ---------------- */
	
	public String reformatText (String text, boolean input) { 
		if (scriptEngine == null) return text;
		String function = input ?  "pre" : "post";
		try {
			Invocable inv = (Invocable) scriptEngine;
			Object eval = inv.invokeFunction(function, text);
			logMessage ("javascript", function + "("+ text + ") evaluated as " + eval);
			if (eval == null) return text;
			return eval.toString();
		} catch (Exception e) {
			logMessage ("javascript", function + "("+ text + ") not evaluated : " + e);
			return text;
		}
	}
	
	/* ------------------ IExternalMemory ---------------*/

	public String getProviderName() {
		return providerName;
	}

	public String getMemoryName() {
		if (name == null) return getProviderName();
		return name;
	}
	
	/* ------------ Log capability ---------- */
	
	public void logMessage (String condition, String message) {
		if (logDest == null) return;
		if (! logList.contains(condition)) return;
		message = SHORT_LOG_DATE_FORMAT.format (new java.util.Date()) + " - <" + condition + "> " + message;
		if (logDest == System.err) // other modules may fill this log file
			message = "[" + getProviderName() + "/" + getMemoryName() + "] " + message;
		else 
			if ("".equals(condition)) System.err.println("[" + getProviderName() + "/" + getMemoryName() + "]: " + message);
		logDest.println (message);
	}
	
	public void logException (Exception e) {
		logMessage ("", "Exception:");
		e.printStackTrace();
		if (logDest != System.err) e.printStackTrace (logDest);
	}
	
	public abstract Object entryToLogString (T entry, boolean detailed, boolean digest);
	
	public interface DigestAlgorithm {
		public Object digest(String source);
	}
	
	private DigestAlgorithm digestAlgo;
	
	public Object digest(String source) {
		if (digestAlgo == null) return "";
		return digestAlgo.digest(source);
	}
	
	public void logEntry (String condition, T entry, boolean detailed) {
		if (logDest == null) return;
		if (! logList.contains(condition)) return;
		logDest.println(entryToLogString(entry, detailed, digestAlgo != null));
	}

	/* ------------ Utility methods ------------- */
	
	private static final Pattern doubleWord = Pattern.compile("(?:^|\\b)(\\w+)\\s(.+?)\\s\\1(?:\\b|$)", Pattern.CASE_INSENSITIVE);
	private int limitReduce = 0;
	
	/// Help method to reduce size of the given string. Seems that Postgres's trigrams are not affected but query is faster
	public String reduceWords (String st) {
		if (limitReduce > st.length()) return st;
		logMessage("reduce-words", "String to reduce(" + st.length() + "): '" + st + "'");
		st = st.toLowerCase(); // ensure The = the
		st = st.replaceAll ("\\W+", " "); // trigrams only include letters, digits and spaces
		Matcher m;
		while ((m = doubleWord.matcher(st)).find()) st = m.replaceAll(" $1 $2 "); 
		st = st.trim();
		logMessage("reduce-words", "String reduced(" + st.length() + "):   '" + st + "'");
		return st;
	}
	
}