/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2013-2016 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat;

import org.omegat.util.Preferences;

import java.util.*;
import java.sql.*;

public abstract class PostgresqlCyclotis<T> extends Cyclotis<T> {
	
	private static final Map<String,Connection> pool = new HashMap<String,Connection>();
	
	protected Connection connection;
	protected PreparedStatement pSelect, pInsert;
	protected String tableRead, tableWrite;
	protected Object mem_id = null;
	protected boolean hasNote = false;
	private String metaInfoTableName;
	
	public PostgresqlCyclotis (Properties propList) throws SQLException, ClassNotFoundException {
		super(propList); // fill name, log	
	
		String driver = propList.getProperty ("jdbc.driver"); if (driver == null) driver = "org.postgresql.Driver";
		else if (driver.contains(",")) { 		// for proxy drivers
			String[] drivers = driver.split(",");
			for (int i = 0; i < drivers.length - 1; i++) Class.forName(drivers[i]);
			driver = drivers[drivers.length - 1];
		}
		Class.forName(driver);
		
		String user = propList.getProperty ("jdbc.user"); if (user == null) user = propList.getProperty ("user");
		String pass = propList.getProperty ("jdbc.password"); if (pass == null) pass = propList.getProperty ("password");
		String jdbc = propList.getProperty ("jdbc.url"); if (jdbc == null) jdbc = propList.getProperty ("jdbc");
		if (jdbc == null) { // Alternative : set jdbc parameters separate
			String host = propList.getProperty("jdbc.host"), port = propList.getProperty("jdbc.port"), db = propList.getProperty("jdbc.db");
			if (host == null) host = "localhost"; if (port != null) host += ":" + port;
			jdbc = "jdbc:postgresql://" + host + "/" + db; // only for Postgresql driver
			if ("true".equals(propList.getProperty("jdbc.ssl"))) jdbc += "?ssl=true";
		}
		this.connection = pool.get(jdbc); 
		if ((this.connection == null) || this.connection.isClosed()) pool.put(jdbc, this.connection = DriverManager.getConnection(jdbc, user, pass));
		
		this.tableWrite = propList.getProperty ("table"); 
		if (tableWrite == null) {
			this.tableWrite = propList.getProperty ("table.write"); 
			if (tableWrite == null) tableWrite = "MEM";
		}
		this.tableRead = propList.getProperty ("table.read"); if (tableRead == null) tableRead = tableWrite;
		
		this.metaInfoTableName = propList.getProperty("table.meta");
		
		logMessage("", "Connection to " + name + " as " + jdbc + " OK.");
		
		// Search for properties and context column types
		PreparedStatement pColumnInfo = connection.prepareStatement ("select column_name, udt_name, udt_schema, character_maximum_length, column_default from INFORMATION_SCHEMA.COLUMNS where table_name = ? and table_schema = ? and table_catalog = current_catalog");
		if (tableWrite.contains(".")) { pColumnInfo.setString(1, tableWrite.substring(tableWrite.indexOf(".") + 1).toLowerCase()); pColumnInfo.setString(2, tableWrite.substring(0, tableWrite.indexOf(".")).toLowerCase()); }
		else { pColumnInfo.setString (1,tableWrite.toLowerCase()); pColumnInfo.setString(2, "public"); }
		ResultSet set = pColumnInfo.executeQuery();
		while (set.next()) this.registerColumn (set, propList);
		pColumnInfo.close();		
	}
	
	protected PostgresqlCyclotis(PostgresqlCyclotis ori) {
		super(ori);
		this.connection = ori.connection; 
		this.tableRead = ori.tableRead; this.mem_id = ori.mem_id; this.hasNote = ori.hasNote; this.metaInfoTableName = ori.metaInfoTableName;
		// do NOT copy SQL statements
	}
	
	protected void registerColumn(ResultSet set, Properties propList) throws SQLException {
		String colName = set.getString("column_name").toUpperCase();
		if (colName.equalsIgnoreCase("NOTE")) this.hasNote = true;
		if (colName.toUpperCase().startsWith("MEM_")) 
			try {
				if (! tableRead.equalsIgnoreCase(tableWrite)) {
					logMessage ("", "We use inheritance, so we must store " + colName);
					throw new Exception(); // go to catch
				}
				if (colName.equalsIgnoreCase("MEM_ID")) {
					int fix = Integer.parseInt (set.getString("column_default"));
					logMessage ("", "Table " + this.tableWrite + " has MEM_ID with default " + fix + ": not saved");
				}
				else if (colName.equalsIgnoreCase("MEM_CODE")) {
					long fix = Long.parseLong (set.getString("column_default"));
					logMessage ("", "Table " + this.tableWrite + " has MEM_ID with default " + fix + ": not saved");
				}
				else if (colName.equalsIgnoreCase("MEM_PATH")) {
					String col_default = set.getString("column_default");
					if ((col_default != null) && (col_default.length() > 0))
						logMessage ("", "Table " + this.tableWrite + " has MEM_PATH with default " + col_default + ": not saved");
					else
						throw new Exception();
				}
			} catch (Exception e) { // Now we must read info from table meta_info
				ResultSet set1 = getInfoResultSet(colName);
				while (set1.next()) {
					if (colName.equalsIgnoreCase("MEM_ID")) this.mem_id = set1.getInt (1);
					else if (colName.equalsIgnoreCase("MEM_CODE")) this.mem_id = set1.getLong (1);
					else if (colName.equalsIgnoreCase("MEM_PATH")) this.mem_id = set1.getString (1);
					logMessage("", "Table " + this.tableWrite + " has " + colName + ", declared as " + this.mem_id);
				}
			}
	}
	
	protected ResultSet getInfoResultSet (String name) throws SQLException {
		if (metaInfoTableName == null) metaInfoTableName = findInfoTableName();
		PreparedStatement pId = connection.prepareStatement ("select " + name + " from public.meta_info where table_name = ? and table_schema = ?");
		if (this.tableWrite.contains(".")) {
			pId.setString (1, this.tableWrite.substring(this.tableWrite.indexOf('.') + 1)); // table_name
			pId.setString (2, this.tableWrite.substring(0,this.tableWrite.indexOf('.'))); // table_schema
		} else {
			pId.setString (1, this.tableWrite); // table_name
			pId.setString (2, "public"); // table_schema			
		}
		return pId.executeQuery();		
	}
	
	private String findInfoTableName() throws SQLException {
		Statement stmt = connection.createStatement();
		try {
			ResultSet set = stmt.executeQuery ("select table_name from information_schema.tables where table_schema  = 'public'");
			while (set.next()) 
				if (set.getString(1).equalsIgnoreCase("meta_view")) return "public.meta_view";
				else if (set.getString(1).equalsIgnoreCase("meta_info")) return "public.meta_info";
				else if (set.getString(1).equalsIgnoreCase("options")) return "public.options";			
		} finally {
			stmt.close();
		}
		throw new SQLException ("Did not find table for meta info"); // only if none of 'return' statements did work
	}
	
	/* ------------------ READ ------------------- */
	
	protected abstract T buildEntry (ResultSet set) throws SQLException;
	
	protected List<T> retreiveQuery (PreparedStatement query) throws SQLException {
		List<T> theList = new LinkedList<T> ();
		try {
			ResultSet set = query.executeQuery();
			while (set.next()) try { theList.add (this.buildEntry (set)); } catch (SQLException sqlIn) { logMessage("", "While retreiving result:"); logException(sqlIn); /* continue; */ }
			set.close();
		} catch (SQLException sqlOut) {
			logMessage("", "After retreiving result:"); 
			logException(sqlOut); // and continue.
			if (theList.isEmpty()) throw sqlOut;
		}
		logMessage("search", "returned " + theList.size() + " entries");
		return theList;
	}
	
	/** Alternative for very big queries **/
	protected Iterable<T> iterateQuery (final PreparedStatement stmt) throws SQLException {
		return new Iterable<T>() {
			public Iterator<T> iterator() {
				try {
					return new Iterator<T>() {
						private boolean didNext = false, hasNext = false;
						private ResultSet set = stmt.executeQuery();
						
						public T next() {
							try {
								if (!didNext) set.next(); didNext = false;
								return buildEntry (set); 
							} catch (SQLException sqlEx) {
								logException (sqlEx);
								return null;
							}
						}
						
						public boolean hasNext() {
							try {
								 if (!didNext) { hasNext = set.next(); didNext = true; }
								 if (!hasNext) set.close();
							} catch (SQLException sqlEx) {
								logException (sqlEx);
							} finally {
								 return hasNext;																
							}
						}
					};
				} catch (Exception e) {
					logException(e);
					return Collections.emptyIterator();
				}
			}
		};
	}

	/* ------------------ SQL SELECT -------------------- */
	
	protected abstract String selectCondition();
	
	/** Beginning of any select expression, without the where clause **/
	protected final String getSelectFrom() {
		if ((! tableRead.equals(tableWrite)) && (mem_id != null))
			if (mem_id instanceof Integer) // mem_id : must join with meta_info
				return "select *, concat(m.table_schema,'.',m.table_name) as mem_name "
					+ "  from " + tableRead + " t left join " + metaInfoTableName + " m on t.mem_id = m.mem_id";
			else if (mem_id instanceof Long) // mem_code : same as previous, except that this is a long
				return "select *, concat(m.table_schema,'.',m.table_name) as mem_name "
					+ "  from " + tableRead + " t left join " + metaInfoTableName + " m on t.mem_code = m.mem_code";
			else if (mem_id instanceof String) // mem_path : end of the path is table name
				return "select *, substring(MEM_PATH from E'/([^/]+?)$') as mem_name from " + tableRead; 
		return "select * from " + tableRead;
	}
	
	protected final PreparedStatement getSelectStatement() throws SQLException {
		if (pSelect == null) {
			String sql = getSelectFrom() + " where " + selectCondition();
			logMessage("sql", "Select statement = " + sql);
			pSelect = connection.prepareStatement (sql);
		}
		return pSelect;
	}

	/* ------------------ SQL INSERT ---------------*/	
	
	protected abstract List<String> insertFields();
	
	protected String getInsertFieldType (String name) { return ""; }
	
	protected PreparedStatement getInsertStatement() throws SQLException {
		if (pInsert == null) { 
			StringBuffer insertBuf = new StringBuffer("insert into ").append(tableWrite).append("(");
			List<String> fields = this.insertFields();
			for (int i = 0; i < fields.size(); i++) { 
				insertBuf.append(fields.get(i)); 
				if (i < fields.size() - 1) insertBuf.append(","); 
			}
			insertBuf.append(") values (");
			for (int i = 0; i < fields.size(); i++) { 
				insertBuf.append("?");
				String type = getInsertFieldType (fields.get(i)); if (! type.equals("")) insertBuf.append("::").append(type);
				if (i < fields.size() - 1) insertBuf.append(",");
			}
			insertBuf.append(")");
			logMessage("sql", "Insert statement = " + insertBuf);
			pInsert = connection.prepareStatement (insertBuf.toString());
			if (this.mem_id != null) 
				if (this.mem_id instanceof Integer) pInsert.setInt (fields.size(), ((Integer) this.mem_id).intValue());
				else if (this.mem_id instanceof Long) pInsert.setLong (fields.size(), ((Long) this.mem_id).longValue());
				else if (this.mem_id instanceof String) pInsert.setString (fields.size(), this.mem_id.toString());
		}
		return pInsert;
	}
		
}