/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2013-2017 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat.glos;

import org.silvestrislab.cyclotis.omegat.PostgresqlCyclotis;

import org.omegat.core.glossaries.IWritableGlossary;
import org.omegat.gui.glossary.GlossaryEntry;
import org.omegat.util.Language;

import java.util.regex.*;
import java.util.*;
import java.sql.*;
				 
public class PostgresqlGlossary extends PostgresqlCyclotis<GlossaryEntry> implements IWritableGlossary {

	private String srcLang;
	private boolean priority = false;
	private Map<String,String> indexTypes = new HashMap<String,String>();
		
	public PostgresqlGlossary (Properties propList) throws SQLException, ClassNotFoundException {
		super (propList);
		
		for (String propName: propList.stringPropertyNames())
			if (propName.startsWith("dict.")) indexTypes.put(propName.substring(5), propList.getProperty(propName));

		priority = "true".equals(propList.getProperty("##PriorityGlossary##"));
		srcLang = propList.getProperty ("srcLang");
		if (srcLang != null) srcLang = postgresIndexType(srcLang);
	}
	
	@Override
	protected GlossaryEntry buildEntry (ResultSet set) throws SQLException {
		String note = set.getString("NOTE"); if (note == null) note = ""; // change in OmegaT 3.5.1
		return new GlossaryEntry (reformatText(set.getString("SRC"),true), reformatText(set.getString("TRA"),true), note, priority, getMemoryName());
	}
	
	protected String selectCondition() {
		logMessage("sql", "Select statement = select * from " + tableRead 
			+ " where to_tsvector(? :: regconfig,src) @@ to_tsquery(? :: regconfig, ?)");
		return " to_tsvector(? :: regconfig,src) @@ to_tsquery(? :: regconfig, ?)";
	}
	
	
	protected List<String> insertFields() {
		List<String> res = new ArrayList<String>(4);
		res.add("SRC"); res.add("TRA"); //res.add("AUTHOR");
		if (this.hasNote) res.add("NOTE");
		if (this.mem_id != null) 
			if (this.mem_id instanceof Integer) res.add("MEM_ID");
			else if (this.mem_id instanceof Long) res.add("MEM_CODE");
			else if (this.mem_id instanceof String) res.add("MEM_PATH");
		return res;
	}
	
	private String postgresIndexType (String langName) {
		if (langName != null) {
			if (langName.startsWith("en")) return "english";
			if (langName.startsWith("fr")) return "french";
			if (langName.startsWith("es")) return "spanish";
			if (langName.startsWith("de")) return "german";
			if (langName.startsWith("da")) return "danish";
			if (langName.startsWith("nl")) return "dutch";
			if (langName.startsWith("fi")) return "finnish";
			if (langName.startsWith("hu")) return "hungarian";
			if (langName.startsWith("it")) return "italian";
			if (langName.startsWith("nb")) return "norwegian";
			if (langName.startsWith("nn")) return "norwegian";
			if (langName.startsWith("no")) return "norwegian";
			if (langName.startsWith("pt")) return "portugese";
			if (langName.startsWith("ro")) return "romanian";
			if (langName.startsWith("ru")) return "russian";
			if (langName.startsWith("sv")) return "swedish";
			if (langName.startsWith("tr")) return "turkish";			
		}
		return "simple";
	}
	
	/* ----------------------- IGlossary ---------------------- */
	
	private static final Pattern PTN_WORD = Pattern.compile("\\pL+");
	
	public List<GlossaryEntry> search(Language sLang, Language tLang, String srcText) throws Exception {
		logMessage("search", "Searching for '" + srcText + "'");
		
		Matcher matcher = PTN_WORD.matcher (srcText);		
		StringBuffer query = new StringBuffer();
		while (matcher.find ()) query.append(" | ").append(matcher.group());
		
		String indexType = srcLang != null ? srcLang : postgresIndexType(sLang.getLanguageCode());

		PreparedStatement pSelect = getSelectStatement();
		pSelect.setString (1, indexType); pSelect.setString (2, indexType);		
		pSelect.setString (3, query.toString().substring(3));		
		return retreiveQuery (pSelect);
	}
	
	/* ----------------------- IWritableGlossary ---------------------- */
	
	public void addEntry(Language sLang, Language tLang, GlossaryEntry entry) throws Exception {
		PreparedStatement pInsert = getInsertStatement();
		try {
			pInsert.setString (1, reformatText(entry.getSrcText(),false)); 
			pInsert.setString (2, reformatText(entry.getLocText(),false));
			if (this.hasNote) pInsert.setString (3, entry.getCommentText());
			pInsert.executeUpdate();
			logEntry ("update-contents", entry, false);
		} catch (Exception e) {
			logException(e); throw e;
		}
	}	
	
	/* ------------  Logging -------------- */
	
	public StringBuffer entryToLogString (GlossaryEntry entry, boolean detailed, boolean digest) {
		StringBuffer buf;
		buf = new StringBuffer("\t<Glossary Entry>\n");
		if (!digest) {
			buf.append ("\t\t<source>" + entry.getSrcText() + "</source>\n");		
			buf.append ("\t\t<target>" + entry.getLocText() + "</target>\n");
		} else {
			buf.append ("\t\t<source digest='" + digest(entry.getSrcText()) + "'>" + entry.getSrcText() + "</source>\n");		
			buf.append ("\t\t<target digest='" + digest(entry.getLocText()) + "'>" + entry.getLocText() + "</target>\n");			
		}
		buf.append ("\t</Glossary Entry>");
		return buf;
	}
	
}
