/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2015-2017 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat.http;

import org.silvestrislab.cyclotis.omegat.HttpCyclotis;
import java.io.*;
import java.util.*;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;

import org.omegat.core.data.*;
import org.omegat.core.matching.external.ProjectMemory.ContextTMXEntry;
import org.omegat.gui.glossary.GlossaryEntry;
import org.omegat.util.Language;
import org.omegat.util.TMXProp;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.events.*;
import static javax.xml.stream.XMLStreamConstants.*;


public class TMXFormat implements HttpOutputFormat {
	public TMXFormat(String name) {
		int idx = name.indexOf("dateFormat");
		if (idx > 0) {
			name = name.substring(idx) + 11;
			if ((idx = name.indexOf(";")) > 0) name = name.substring(0, idx);
			if ((idx = name.indexOf("!")) > 0) {
				String lang = name.substring(0, idx), spec = name.substring(idx + 1);
				tmxDateFormat = new SimpleDateFormat(spec, new java.util.Locale(lang));
			}
		}
	}
	
	public String MIME() { return "text/xml"; }
	
	private DateFormat tmxDateFormat = new SimpleDateFormat("yyyyMMdd'T'HHmmss'Z'");
	
	class TmxHandler<T> {
		ProjectProperties prop = org.omegat.core.Core.getProject().getProjectProperties();
		Language srcLang = prop.getSourceLanguage(), traLang = prop.getTargetLanguage();
			
		//tmxDateFormat.setTimeZone(TimeZone.getTimeZone("GMT+00:00"));
		
		String lang = "", source = "", target = "", note = ""; List<TMXProp> tmxAttributes = new ArrayList<TMXProp>();
        String author = "", changer = ""; Date creationDate = new Date(0), changeDate = new Date(0); int score = 100; String context = "";
		
		final HttpCyclotis<T> cyclotis; 
		public TmxHandler(final HttpCyclotis<T> cyclotis) { this.cyclotis = cyclotis; }
		
		public void startElement(StartElement el) {
			String localName = el.getName().getLocalPart();
			if (localName.startsWith("tu")) {
                author = el.getAttributeByName(new QName("creationid")).getValue();
                creationDate = parseDate(el.getAttributeByName(new QName("creationdate")).getValue());
                changer = el.getAttributeByName(new QName("changeid")).getValue();
                changeDate = parseDate(el.getAttributeByName(new QName("changedate")).getValue());
             }
             if (localName.equals("tuv")) {
                lang = el.getAttributeByName(new QName("", "lang", "xml")).getValue(); 
                if ((lang == null) || (lang.length() == 0)) { 
                    if (target.length() == 0) lang = "x-src";
                    if (source.length() == 0) lang = "x-tra";
                }
				if (lang.equalsIgnoreCase(srcLang.getLanguage())) lang = "x-src";
				if (lang.equalsIgnoreCase(srcLang.getLanguageCode())) lang = "x-src";
				if (lang.equalsIgnoreCase(srcLang.getLocaleCode())) lang = "x-src";
				if (lang.equalsIgnoreCase(traLang.getLanguage())) lang = "x-tra";
				if (lang.equalsIgnoreCase(traLang.getLanguageCode())) lang = "x-tra";
				if (lang.equalsIgnoreCase(traLang.getLocaleCode())) lang = "x-tra";					
            }
			if (localName.equalsIgnoreCase("note")) lang = "note:";
            if (localName.equalsIgnoreCase("prop")) lang = "prop:" + el.getAttributeByName(new QName("type")).getValue();				
		}
		
		private Date parseDate (String txt) {
			try { return tmxDateFormat.parse(txt); }
			catch (Exception e1) {
				tmxDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", java.util.Locale.ENGLISH);
				try { return tmxDateFormat.parse(txt); }
				catch (Exception e2) { return new Date(0); }
			}
		}
		
		public void characters(String data) {
			if (lang.equalsIgnoreCase("x-src")) source += data;
			if (lang.equalsIgnoreCase("x-tra")) target += data;
			if (lang.equalsIgnoreCase("note:")) note += data;
			if (lang.startsWith("prop:")) tmxAttributes.add (new TMXProp (lang.substring(5),data));
			if (lang.equalsIgnoreCase("prop:Cyclotis:score")) try { score = (int)(100.0 * Integer.parseInt(data)); } catch (Exception e) { }
			if (lang.equalsIgnoreCase("prop:Cyclotis:mem_name")) tmxAttributes.add (new TMXProp ("Row.Table", data));
		}
		
		public void fillEntry (PrepareTMXEntry entry) {
			try {
				entry.source = cyclotis.reformatText(source,true);
				entry.translation = cyclotis.reformatText(target,true);
				entry.creator = author;
				entry.creationDate = creationDate.getTime();
				entry.changer = changer;
				entry.changeDate = changeDate.getTime();
				entry.note = note;
				entry.otherProperties = tmxAttributes;
			} catch (Exception e) {
				// Properties not mandatory
			}			
		}
		
		public T next (XMLEventReader xmlEventReader) throws XMLStreamException {
			while (xmlEventReader.hasNext()) {
				XMLEvent current = xmlEventReader.nextEvent();
				switch (current.getEventType()) {
					case START_ELEMENT:
						try {
							StartElement el = (StartElement) current;
							cyclotis.logMessage ("http-response", "HTTP-XML: start " + el.getName());
							this.startElement(el);
						} catch (ClassCastException e) {
							
						}
						break;
					case CHARACTERS:
						try {
							Characters el = (Characters) current;
							cyclotis.logMessage ("http-response", "HTTP-XML: characters " + el.getData());
							this.characters(el.getData());
						} catch (ClassCastException e) {
							
						}
						break;
					case END_ELEMENT:
						try {
							EndElement el = (EndElement) current;
							cyclotis.logMessage ("http-response", "HTTP-XML: end " + el.getName());
							if (el.getName().getLocalPart().equals("tu")) return cyclotis.analyzeLine (TMXFormat.this, this);
							if (el.getName().getLocalPart().equals("body")) return null;
						} catch (ClassCastException e) {
							
						}
						break;
				}
			}
			return null;
		}
	}
	
	private XMLInputFactory xmlInputFactory = XMLInputFactory.newInstance();
	
	public <T> List<T> readFromGet (final HttpCyclotis<T> cyclotis, InputStream is, String encoding) throws Exception {
		XMLEventReader xmlEventReader = xmlInputFactory.createXMLEventReader(is, encoding);
		List<T> theList = new ArrayList<T>(); TmxHandler<T>  handler = new TmxHandler<T> (cyclotis);
		T next = handler.next(xmlEventReader); while (next != null) theList.add (next);		
		return theList;
	}
	
	public <T> Iterator<T> iterator (final HttpCyclotis<T> cyclotis, final InputStream is, final String encoding) {
		try {
			final XMLEventReader xmlEventReader = xmlInputFactory.createXMLEventReader(is, encoding);
			return new Iterator<T>() {
				private final TmxHandler<T>  handler = new TmxHandler<T> (cyclotis);
				private T current = handler.next(xmlEventReader);
				
				public boolean hasNext() {
					return current != null;
				}
				
				public T next() {
					T prev = current; 
					try { current = handler.next(xmlEventReader); } catch (Exception e) { current = null; }
					return prev;
				}
			};			
		} catch (Exception e) {
			cyclotis.logException(e); return java.util.Collections.emptyIterator();
		}
	}
	
	public PrepareTMXEntry buildTmxEntry (org.silvestrislab.cyclotis.omegat.tm.HttpMemory memory, Object args) {
		PrepareTMXEntry entry = new PrepareTMXEntry();
		TmxHandler handler = (TmxHandler) args;
		handler.fillEntry (entry);
		String mem_name = entry.getPropValue("prop:Cyclotis:mem_name");
		entry.otherProperties = memory.store.fromDatabase (entry.otherProperties, entry);
		if (mem_name != null) entry.otherProperties.add (new TMXProp("db.table.name", mem_name));
		return entry;		
	}
	
	public ContextTMXEntry buildContextEntry (org.silvestrislab.cyclotis.omegat.proj.HttpProjectMemory memory, Object args) {
		ContextTMXEntry entry = new ContextTMXEntry ();
		TmxHandler handler = (TmxHandler) args;
		handler.fillEntry (entry);
		memory.keyStore.fillEntry(entry);
		memory.contextMode.fillEntry (memory, entry, handler.context);

		try { entry.linked = TMXEntry.ExternalLinked.valueOf (entry.getPropValue("externalLinked")); } catch (Exception e2) { }
		return entry;	
	}
	
	public GlossaryEntry buildGlossaryEntry(org.silvestrislab.cyclotis.omegat.Cyclotis memory, Object args, boolean priority) {
		TmxHandler handler = (TmxHandler) args;
		if (handler.note == null) handler.note = ""; // in OmegaT 3.5.1, null may cause a bug
		return new GlossaryEntry (handler.source, handler.target, handler.note, priority, memory.getMemoryName());		
	}
	
}
