/**************************************************************************
 Silvestris Cyclotis
 
 Copyright (C) 2013-2016 Silvestris project (http://www.silvestris-lab.org/)
 
 This file is part of Cyclotis plugin for OmegaT
 
 Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
 You may not use this work except in compliance with the Licence.
 You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

 Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the Licence for the specific language governing permissions and limitations under the Licence. 
 **************************************************************************/

package org.silvestrislab.cyclotis.omegat.proj;

import org.silvestrislab.cyclotis.omegat.PostgresqlCyclotisTMX;
import org.silvestrislab.cyclotis.omegat.proj.ctx.ContextMode;
import org.silvestrislab.cyclotis.omegat.proj.ctx.IntContextMode;
import org.silvestrislab.cyclotis.omegat.proj.ctx.StringContextMode;

import org.omegat.core.matching.external.ProjectMemory;
import org.omegat.core.matching.external.ProjectMemory.ContextTMXEntry;
import org.omegat.core.Core;
import org.omegat.core.data.SourceTextEntry;
import org.omegat.core.data.PrepareTMXEntry;
import org.omegat.util.Preferences;
import org.omegat.util.TMXProp;

import java.util.*;
import java.sql.*;

public class PostgresqlProjectMemory extends PostgresqlCyclotisTMX<ContextTMXEntry> implements ProjectMemory.IConvertibleExternalProjectMemory {

	private java.sql.PreparedStatement pRemove;
	private EntryKeyStore keyStore;
	// server date - client date : ensure that timestamps are correct
	private long timeStampDiff = 0;
	
	public PostgresqlProjectMemory (Properties propList) throws SQLException, ClassNotFoundException {
		super (propList);
			
		try {
			PreparedStatement sNow = connection.prepareStatement("select NOW()");
			ResultSet rs = sNow.executeQuery(); rs.next();
			timeStampDiff = rs.getTimestamp(1).getTime() - System.currentTimeMillis();
			logMessage("", "Server to client timestamp difference: " + timeStampDiff);
		} catch (Exception e) {
			logMessage("", "Cannot find server timestamp : " + e.getMessage());
		}

		if (propList.getProperty("remove.setNull") != null) // user explicitely decided to use rule or not (if not specified, call the server)
			pRemove = createRemoveRule("true".equals(propList.getProperty("remove.setNull")));
		
		if (this.col_props != null)	{ keyStore = new EntryKeyStore (propList); logMessage("struct", "Storing properties as " + keyStore); }
		else logMessage("struct", "Table has no properties, do not store key");
	}
		
	public org.omegat.core.matching.external.IExternalMemory asTranslationMemory() {
		return new org.silvestrislab.cyclotis.omegat.tm.PostgresqlMemory(this) {
			protected PrepareTMXEntry buildEntry (ResultSet set) throws SQLException {
				ContextTMXEntry ctx = PostgresqlProjectMemory.this.buildEntry(set);
				if (ctx.entryNum >= 0) throw new SQLException("Not an orphan");
				return super.buildEntry (set);
			}
		};
	}

	@Override
	protected List<String> insertFields() {
		List<String> res = new ArrayList<String>(4);
		res.add("SRC"); res.add("TRA"); res.add("AUTHOR");
		res.add("CONTEXT");
		if (this.hasNote) res.add("NOTE");
		if (this.col_props != null) res.add("PROPS");
		if (this.mem_id != null) 
			if (this.mem_id instanceof Integer) res.add("MEM_ID");
			if (this.mem_id instanceof Long) res.add("MEM_CODE");
			else if (this.mem_id instanceof String) res.add("MEM_PATH");
		return res;
	}
	
	/* ------------------ SQL --------------- */
	
	protected String selectCondition() { return "date >= ? or changedate >= ?"; }
	
	protected ContextTMXEntry buildEntry (ResultSet set) throws SQLException {
		ContextTMXEntry entry = new ContextTMXEntry ();
		entry.source =  reformatText(set.getString("SRC"),true);
		entry.translation = reformatText(set.getString("TRA"),true);
		try {
			entry.creator = set.getString("AUTHOR");
			entry.creationDate = set.getTimestamp("DATE").getTime();
			entry.changer = set.getString("CHANGER");
			entry.changeDate = set.getTimestamp("CHANGEDATE").getTime();
			entry.note = set.getString("NOTE");
		} catch (Exception e) {
			// Properties not mandatory
		}
		entry.otherProperties = this.col_props.read(set.getObject("PROPS"));
		if (keyStore != null) keyStore.fillEntry(entry);
		contextMode.fillEntry (this, entry, set);
		return entry;
	}
	
	/* ------------------ ProjectMemory.IExternalProjectMemory ---------------*/
	
	public Iterable<ContextTMXEntry> findChanges (long timeStamp) throws Exception {
		Timestamp date = new java.sql.Timestamp (timeStamp + timeStampDiff); 
		logMessage("search", "Searching for segments after " + LONG_LOG_DATE_FORMAT.format(date));
		PreparedStatement pSelect = getSelectStatement();
		pSelect.setTimestamp (1, date); pSelect.setTimestamp (2, date);
		if (timeStamp < 1E9) return iterateQuery(pSelect); else return retreiveQuery(pSelect);
	}
	
	public void registerTranslation(ContextTMXEntry entry) throws Exception {
		logMessage("update", "Writing entry '" + entry.source + "'");
		// Fill CONTEXT column. 
		// We cannot use entry.sourceEntry.entryNum() because two entries with same key must not be stored separately
		// So, we could store the entire key, as a string
		// but since the field is mostly used for comparison, it should be as small as possible,
		// so we use the hashCode, with the hope it is unique in the entire project.
		if (this.pInsert == null) {
			this.pInsert = getInsertStatement();			
			pInsert.setString (3, Preferences.getPreferenceDefault(Preferences.TEAM_AUTHOR, System.getProperty("user.name")));		// set author only once, will never change
		}
		SourceTextEntry ste = Core.getProject().getAllEntries().get (entry.entryNum - 1);
		contextMode.setStatementValue (pInsert, 4, ste, entry.isDefault);
		if (this.hasNote) getInsertStatement().setString(5, entry.note);
		// Entry number is stored, but as a property: not used for comparison.
		if (this.col_props != null) {
			List<TMXProp> list = store.toDatabase(entry, ste);
			if ((keyStore != null) && (keyStore.shouldWrite(entry))) list.addAll(keyStore.buildProperties(entry));
			if (entry.linked != null) list.add (new TMXProp("externalLinked", entry.linked.toString()));
			pInsert.setObject(this.hasNote ? 6 : 5, col_props.write (list));
		}
		super.registerTranslation (entry);
	}
	
	/** Create rule for remove: may set translation to null instead (forced = false). **/
	private PreparedStatement createRemoveRule(Boolean forced) throws SQLException {
		if (forced == null) 
			try {	// try to ask the server for rule
				logMessage("struct", "Trying to ask the server whenever it has a deletion rule or not");
				PreparedStatement pFindDelRule = connection.prepareStatement("select definition from pg_rules where schemaname=? and tablename=?");
				if (this.tableWrite.contains(".")) { pFindDelRule.setString(1, this.tableWrite.substring(0, this.tableWrite.indexOf('.'))); pFindDelRule.setString(2, this.tableWrite.substring(1 + this.tableWrite.indexOf('.')));  }
				else { pFindDelRule.setString(1, "public"); pFindDelRule.setString(2, this.tableWrite); }			
				ResultSet RS = pFindDelRule.executeQuery();
				boolean hasDelRule = false; while (RS.next()) if (RS.getString("definition").toUpperCase().contains(" ON DELETE ")) hasDelRule = true;
				forced = ! hasDelRule;
				logMessage("struct", "Detected that the server " + (hasDelRule ? " has " : " does not have ") + " a deletion rule");
			} catch (Exception e) {
				logMessage("struct", "Could not ask the server for deletion rule: we do as if it does not have it");
				forced = true;
			}
		if (! forced) // forced to false or a rule has been found
			return connection.prepareStatement ("delete from " + tableWrite + " where context=?"
				+ (contextMode.isUnique() ? "" : " and src=?"));			
		else	// forced to true or no rule has been found
			return connection.prepareStatement ("update " + tableWrite + " set tra=null where context=?"
				+ (contextMode.isUnique() ? "" : " and src=?"));			
	}

	public void removeTranslation(ContextTMXEntry entry) throws Exception {
		SourceTextEntry ste = Core.getProject().getAllEntries().get (entry.entryNum - 1);
		logMessage("update", "Removing entry '" + entry.source + "', id = " + entry.entryNum);
		try {
			if (pRemove == null) pRemove = createRemoveRule(null); // ask server for correct rule
			contextMode.setStatementValue (pRemove, 1, ste, entry.isDefault);
			if (! contextMode.isUnique()) pRemove.setString (2, entry.source);
			pRemove.executeUpdate ();
		} catch (Exception e) {
			logException (e); throw e;
		}
	}
	
	
}