module Spongiae
    File.open("#{File.dirname(__FILE__)}/HISTORY",'r') do |f|
        VERSION = $1 if f.gets =~ /^(\d+\.\d+\.\d+)/
    end
end

require 'anguilla'    
require 'spongiae/unit'

##
# TMX module for Spongiae
# Not a format because producer does not use same translate method as for unilingual formats

module Spongiae::TMX
    
    ##
    # Gets only meta-info about a file, reading beginning
    # use regexes because anguilla parser would read the full file
    def self.read_meta(file,encoding = 'UTF-8')
        res = {}
        File.open(file, "r:#{encoding}") do |f|
            while line = f.gets
                res['srcLang'] = $1 if line =~ /\ssrclang\s*=\s*["'](.+?)["']/
                return res if (res['srcLang'] != nil) 
            end
        end
    end
       
    class TmxCallbacks
        include REXML::StreamListener
        
        def initialize(sub)
            @sub = sub
            @lastId = 0
            @creationtool = ''
        end
        
        def tag_start(element, attributes)
            if element == 'header'
                @creationtool = attributes['creationtool']
                @srcLang = attributes['srclang']
                require 'digest/sha1' if @creationtool == 'OmegaT'
            elsif element == 'tu'
               if attributes['tuid'] != nil then
                  @tu = Spongiae::Unit::Multilingual.new(attributes, {})
               else
                  @tu = Spongiae::Unit::Multilingual.new(@lastId, {})
                  @lastId = @lastId + 1
               end
               @tu.props[:srcLang] = @srcLang if @creationtool == 'OmegaT' 
               @lang = nil; @where = ''
               @attr = attributes
            elsif element == 'tuv'
               @lang = attributes['xml:lang']
               if @lang == '' or @lang == nil then 
                  @lang = attributes['lang'] 
               end
               @langProps = {}
               @attr = attributes
            elsif element == 'seg'
               @text = ''; @prop = nil
            elsif element == 'prop'
               @text = ''; @prop = attributes['type']
            end	
        end
        
        def text(text) 
            @text = @text + text if @text != nil
        end
        
        def tag_end(element)
            if element == 'tu'
                @tu.rebuild_id_omegat! if @creationtool == 'OmegaT' 
                @sub.call(@tu)
            elsif element == 'tuv'
               @lang = nil
            elsif element == 'seg'
               @tu.variants[@lang] = Spongiae::Unit::Variant.new(@attr, @langProps, @text)
            elsif element == 'prop'
               if @lang != nil then @langProps[@prop] = @text elsif @tu != nil then @tu.props[@prop] = @text end
            end
        end    
    end # TmxCallbacks
    
    class TmxReader      
        def initialize(file, props = {})
            @file = file
        end
        
        def read_unit(&sub)
            Anguilla::parse(@file,TmxCallbacks.new(sub))
        end
    end
    
    ##
    # Takes result from Spongiae::XLIFF::read_units from various languages and merges them to TMX units
    # Parameter xliff_map is a hash { target language => { segment_id => bilingual unit } }
    def self.merge(src_lang, xliff_map, &sub)
        firstLang, firstFile = xliff_map.first
        xliff_map.delete firstLang
        
        firstFile.each do |unitId, blUnit|
            tu = Spongiae::Unit::Multilingual.new(unitId, blUnit.props)
            if blUnit.is_a? Spongiae::Unit::Bilingual then  # XLIFF
                tu.variants[src_lang] = Spongiae::Unit::Variant.new(src_lang, nil, blUnit.srcText)
                tu.variants[firstLang] = Spongiae::Unit::Variant.new(firstLang, nil, blUnit.traText)
            elsif blUnit.is_a? Spongiae::Unit::Multilingual then # TMX
                blUnit.variants.each { |k,v| tu.variants[k] = v }
            end
            xliff_map.each do |lang, langMap|
                if langMap[unitId].is_a? Spongiae::Unit::Bilingual then # XLIFF
                    tu.variants[lang] = Spongiae::Unit::Variant.new(lang, nil, langMap[unitId].traText) if langMap[unitId] != nil
                elsif langMap[unitId].is_a? Spongiae::Unit::Multilingual then # TMX
                    langMap[unitId].variants.each { |k,v| tu.variants[k] = v }
                end
                langMap.delete unitId   # for the recursive call, only units not yet merged
            end
            yield tu
        end
        self.merge(src_lang,xliff_map) { |tu| sub.call(tu) } if xliff_map.count > 0     # with units not present in first file
    end
    
    class TmxWriter
        
        def initialize(target,attr = { :tool => 'Spongiae', :version => Spongiae::VERSION })
            @target = target
            target.puts <<"EOF"
<?xml version="1.0" encoding="UTF-8" ?>
<!DOCTYPE tmx SYSTEM "tmx14.dtd">
<tmx version="1.4">
      <header
         creationtool="#{attr[:tool]}"
         creationtoolversion="#{attr[:version]}"
         o-tmf="Spongiae"
         segtype="sentence"
         adminlang="#{if defined? attr[:adminlang] then attr[:adminlang]  else 'EN-US' end}"
         srclang="#{if defined? attr[:srclang] then attr[:srclang]  else 'EN-US' end}"
         datatype="PlainText" />
      <body>
EOF
        end
        
        def add(tuid,tu)
            @target.puts "          <tu #{tu.attr.collect { |k,v| "#{k} = #{v.encode(:xml => :attr)}" }.join(' ') }>\n"
            tu.props.each { |k,v| @target.puts "            <prop type=\"#{k}\">#{v.encode(:xml => :text)}</prop>" } unless tu.props == nil 
            tu.variants.each do |lang, var|
               tuvProps = ''
               var.props.each { |k,v| tuvProps = tuvProps + "\n                <prop type=\"#{k}\">#{v.encode(:xml => :text)}</prop>" } unless var.props == nil 
               @target.puts <<"EOF".gsub(/\n\n/,"\n").gsub(/ >/, '>')
            <tuv #{var.attr.collect { |k,v| "#{k} = #{v.encode(:xml => :attr)}" }.join(' ') }>#{tuvProps}
               <seg>#{var.text.encode(:xml => :text)}</seg>
            </tuv>
EOF
            end
            @target.puts "          </tu>\n"
        end
        
        def comment(text) @target.puts "          <!-- #{text} -->\n" end
            
        def close
            @target.puts "      </body>\n"
            @target.puts "</tmx>\n"
        end
        
    end # TMXWriter

end # Module

