require 'spongiae/unit'
require 'spongiae/tags'

require 'nokogiri'
require 'anguilla/zip'

module Spongiae
   module Formats
       
       class OpenXmlString
           def initialize(text, runs = []) @text = text; @runs = runs end

           def to_native_xml() 
               res = ''
               for i in 0 .. (@runs.count - 1)
                  res << '<w:r>'
                  @runs[i].item.each { |pr| res << pr.to_native_xml }
                  res << '<w:t>'
                  if @runs[i+1] != nil then
                      res << @text[@runs[i].pos .. @runs[i+1].pos].encode(:xml => :text)
                  else
                      res << @text[@runs[i].pos .. -1].encode(:xml => :text)
                  end
                  res << '</w:t></w:r>'
               end
               return res
           end
           def to_xliff(segmented: true) 
               res = ''               
               for i in 0 .. (@runs.count - 1)
                   if i == @runs.count - 1 then stop = @text.length else stop = @runs[i+1].pos end
                   if @runs[i].item.count == 0 then 
                       res << @text[@runs[i].pos .. (stop - 1)].encode(:xml => :text)
                   else
                       if segmented then res << "<bpt id='#{i}' />" else res << "<g id='#{i}'>" end
                       res << @text[@runs[i].pos .. (stop - 1)].encode(:xml => :text)
                       if segmented then res << "<ept id='#{i}' />" else res << '</g>' end
                    end
               end
               return res
           end
           
           # Convert from tagged String
           def self.fromTagged(ts, unit)
               runs = [ Spongiae::Tags::Placeable.new(0, []) ] # Temporary : work always with one run
               # runs = []; txt = ts.text.clone
               #ts.placeables.each do |tag|
                   
               #end
               return OpenXmlString.new(ts.text,runs)
           end
       end
       
       class OpenXmlCallbacks < Nokogiri::XML::SAX::Document
           
           def initialize(file,sub,withTags = true)
               @sub = sub; @file = file; @id = 0; @text = ''
               @runs = [];  @tagStack = []
               @withTags = withTags
           end
           
           def to_hash(attrs)
               res = Hash.new
               attrs.each { |row| res["#{row[1]}:#{row[0]}"] = row[-1] }
               return res
           end
           
           def start_element(name, attrs = []) start_element_namespace(name,attrs) end
               
           def start_element_namespace(name, attrs = [], prefix = nil, uri = nil, ns = [])
               if name == 'p' and prefix == 'w' then
                   @id = @id + 1 
               elsif name == 'r' and prefix == 'w' then
                   @runs << Spongiae::Tags::Placeable.new(@text.length, ['A'])
               elsif @runs.count > 0 and @runs[-1].item[-1] != nil then
                   if name == 't' and prefix == 'w' then
                       @runs[-1].item << nil # stop marker
                   else
                       tag = Spongiae::Tags::MarkupTag.new; tag.type = +1; tag.name = name
                       tag.name = "#{prefix}:#{name}" unless prefix == nil
                       tag.attrs = to_hash(attrs)
                       @tagStack << tag.id = @runs[-1].item.count
                       @runs[-1].item << tag
                   end
               end
           end
           
           def end_element_namespace(name, prefix = nil, uri = nil)
               if name == 'p' and prefix == 'w' then 
                   if not(@withTags) or (@runs.count == 1 and @runs[0].item.count == 0) then
                       @sub.call Spongiae::Unit::Unilingual.new(@file, @id.to_s, nil, @text)
                   else
                       @sub.call Spongiae::Unit::Unilingual.new(@file, @id.to_s, nil, OpenXmlString.new(@text,@runs))
                   end
                   @text = ''; @runs = []
               elsif name == 'r' and prefix == 'w' then
                   @runs[-1].item.unshift   # remove the dummy marker (in beginnig of the run, not in the end!)
               elsif @runs.count > 0 and @runs[-1].item[-1] != nil then
                   tag = Spongiae::Tags::MarkupTag.new; tag.type = -1; tag.name = name
                   tag.name = "#{prefix}:#{name}" unless prefix == nil
                   tag.id = @tagStack.pop
                   @runs[-1].item << tag
               end
           end
           
           def characters(text) 
               if @id != nil then @text = @text + text end
           end
       end
       
       class OpenXmlCallbacksOutput <  OpenXmlCallbacks
           def initialize(dest,file,sub)
               super(file, sub,false); @dest = dest
               @NS = Hash.new
           end
           
           def start_element_namespace(name, attrs = [], prefix = nil, uri = nil, ns = [])
               super(name, attrs, prefix, uri, ns)
               if prefix == nil then @dest.print "<#{name}" else @dest.print "<#{prefix}:#{name}" end
               ns.each do |item| 
                   unless @NS[item[0]] == item[1]
                       @dest.print " xmlns:#{item[0]} = \"#{item[1]}\"" 
                   else
                       @NS[item[0]] = item[1]
                   end
               end
               attrs.each { |item| @dest.print " #{item[1]}:#{item[0]} = \"#{item[-1]}\"" }
               @dest.print ">"               
           end
           
           def end_element_namespace(name, prefix = nil, uri = nil)
               super(name, prefix, uri)
               if prefix == nil then @dest.print "</#{name}>" else @dest.print "</#{prefix}:#{name}>" end
           end
           
           def characters(text) 
               if @id != nil then 
                   @text = @text + text 
               end
           end
       end
       
       
       class OpenXmlFile
           
           def initialize(file, props = {})
               @file = file; @props = props
           end
           
           # read_strings : build unit for each string
           # For Plain Text, this is one string per line.
           def read_unit(&sub)
               Anguilla::Zip.browsingZip(@file) do |entry|
                    if entry.name =~ /(document\d?|footnotes?)\.xml/ then                       
                        callback = OpenXmlCallbacks.new(entry.name,sub, @props['tags'] !~ /no|false/)
                        parser = Nokogiri::XML::SAX::Parser.new(callback)
                        parser.parse(entry.get_input_stream)
                    end
               end
           end
           
           def translate(dest_file_name,translations_map,props={})
               Zip::OutputStream.open(dest_file_name) do |outs|
                   Anguilla::Zip.browsingZip(@file) do |entry|
                        outs.put_next_entry entry.name
                        if entry.name =~ /\/(document\d?|footnotes?)\.xml$/ then                       
                           callback = OpenXmlCallbacksOutput.new(outs, entry.name, Proc.new { |unit|
                                tra = unit.text
                                tra = translations_map["#{unit.file}!!#{unit.id}"] if translations_map["#{unit.file}!!#{unit.id}"] != nil
                                if tra.is_a? String then
                                    outs.print "<w:r><w:t>#{tra.encode(:xml => :text)}</w:t></w:r>"
                                elsif tra.is_a? OpenXmlString then
                                    outs.print tra.to_native_xml
                                elsif tra.is_a? Spongiae::Tags::TaggedString then # read from XLIFF
                                    outs.print OpenXmlString.fromTagged(tra,unit).to_native_xml
                                end 
                            })
                           parser = Nokogiri::XML::SAX::Parser.new(callback)
                           parser.parse(entry.get_input_stream)
                        else
                            outs.copy_raw_entry entry                            
                        end
                   end
               end                   
           end
           
       end
       
   end
end
