#! /usr/bin/env perl

=head1 DESCRIPTION

This HTTP application creates and deletes Cyclotis databases, and can list existing ones. 
This is a REST interface which can answer in XML, JSON or YAML

=head1 INSTALLATION REQUIREMENTS

This script does not replace SQL/db/dbcreator.pl, it is based on it: you must install it and put in the YAML configuration where it is.

If the configuration implies to create catalogs, it must be run on the same machine as the Postgres server.
Else, if all is in the same database, it is enough to call dbcreator.pl once on the Postgres machine, then everything works via DBI.

=cut

use Dancer;
use DBI;
use vr;

$ENV{CYCLOTIS_MODE} = config->{'SQL-scripts'}{mode} if config->{'SQL-scripts'}{mode}; 
push(@main::INC, config->{'SQL-scripts'}{dir});
eval "require $ENV{CYCLOTIS_MODE}::TableCreator;" or die "Cannot find module TableCreator for mode $ENV{CYCLOTIS_MODE} : please install SQL package and set location in config.yml";
require DbCreator or die "Cannot find module DbCreator: please install SQL package and set location in config.yml";
eval sprintf('require formats::%s;', config->{formats}{default}) or die sprintf('Cannot find format %s. Check config.yml and formats/ directory for availability', config->{formats}{default});
$main::SERVER = `hostname`; chomp $main::SERVER;

opendir (FMTDIR, 'formats'); my @FORMATS;
while (my $f = readdir(FMTDIR)) {
	if ($f =~ /\.pm$/) {
		if (require "formats/$f") {
			push(@FORMATS, $1) if $f =~ /^(\w+)\.pm$/;
		}
	}
}
closedir(FMTDIR);
print STDERR "Supported formats : ", join(',',@FORMATS), "\n";

sub add_get {
	my $url = shift; my $method = shift;
	get $url => sub { return &$method("formats::" . config->{formats}{default}); };
	foreach my $fmt (@FORMATS) { get "$url.$fmt" => sub { return &$method("formats::$fmt"); }; } 
}

# 	----------------- Database creation -----------------

=head1 REST API messages

=head2 GET /createdb?projname=xxx&src=en&tra=fr

Creates the table for given project

=cut

add_get '/createdb' => sub {
	my $fmt = shift;

	my %spec = vr::fillDatabaseParams();
	$spec{mem_name} = vr::interpolation(config->{list}{view}{fields}{'mem_name'} || '$table', \%spec);
	
	my $creator = bless { dbiSpec => vr::dbiSpec(\%spec) }, "$ENV{CYCLOTIS_MODE}::TableCreator";
	$creator->{dbh} = DBI->connect($creator->{dbiSpec}, $spec{user}, $spec{pass});
	$creator->loadTables(); eval { $creator->doCreate_table($spec{table}, $spec{parent}); };
	
	if (my $cause = $DBI::errstr || $@) { 
		my $where = $1 if $cause =~ s/(?:\W)at (.+?)$//;	
		if ($cause =~ /already/) {
			my %params = params;
			return $fmt->envelope (['warn' => $cause, $where], $fmt->memory(%params,%spec));
		} else {
			return $fmt->message(error => $cause, $where);
		}
	} else { 
		my %params = params;
		if ($spec{'rule-field'}) {
			DbCreator::doCreate_rule($creator->{dbh}, $spec{table}, $spec{'rule-field'}, $spec{'rule-action'});
			if (my $cause = $DBI::errstr || $@) {
				my $where = $1 if $cause =~ s/(?:\W)at (.+?)$//;
				return $fmt->envelope (['warn' => $cause, $where], $fmt->memory(%params,%spec));
			}
		}
		return $fmt->envelope (['OK'], $fmt->memory(%params,%spec));	
	}
	$creator->{dbh}->disconnect;
};

# 	----------------- Database deletion -----------------

=head2 GET /dropdb?projname=xxx&src=en&tra=fr

Creates the table for given project

=cut

add_get '/dropdb' => sub {
	my $fmt = shift; my $cause;

	my %spec = vr::fillDatabaseParams();
	my $DBI = DBI->connect(vr::dbiSpec(\%spec), $spec{'user'}, $spec{'pass'})
		or $cause = "Cannot connect to DB: $DBI::errstr";
	$DBI->do ("drop table " . $spec{'table'} . ' cascade') unless $cause; $cause = $DBI::errstr;
	if (config->{list}{sql}{table} and not $cause) {
		my ($infoTable, $infoSchemaCol, $infoTableCol) = (config->{list}{sql}{table}, 'table_schema', 'table_name');
		if (ref ($infoTable)) { 
			$infoSchemaCol &&= $infoTable->{fields}{'table_schema'}; $infoTableCol &&= $infoTable->{fields}{'table_name'};
			$infoTable = $infoTable->{name};
		}
		my ($schema, $table) = split(/\./, $spec{'table'}); ($schema, $table) = ('public', $spec{'table'}) unless $table;
		$DBI->do ("delete from $infoTable where lower($infoSchemaCol) = '\L$schema\E' and lower($infoTableCol) = '\L$table\E'\n")
	}
	if ($cause) {
		my $where = $1 if $cause =~ s/(?:\W)at (.+?)$//;	
		return $fmt->message(error => $cause, $where);
	} else {
		return $fmt->message(OK => "Memory $spec{'table'} deleted");
	}
	$DBI->disconnect;
};

# 	----------------- List existing memories -----------------

sub xmlFlatten {
	my $Hash = shift; my $ParamsTab = shift; my $StartKey = shift;
	if (ref($Hash) =~ /ARRAY/) { 
		foreach my $mem (@$Hash) { 
			$mem->{table} ||= sprintf('%s.%s', $mem->{table_schema}, $mem->{table_name}); 
			$mem->{mem_name} = vr::interpolation(config->{def}{mem_name}, $mem) || $mem->{table_name}; 
        } 
		return ($StartKey => $Hash); 
	}
	elsif (ref($Hash) =~ /HASH/) {
		my %RES = (); my $Param0 = shift(@$ParamsTab) or die "Not enough parameters";
		$Param0->{xml} ||= $Param0->{'var-name'}; # compatibility
		while (my ($k, $v) = each (%$Hash)) {
			my %RES1 = xmlFlatten ($v, $ParamsTab, "$StartKey $Param0->{xml}='$k' ");
			%RES = (%RES, %RES1);
		}
		unshift(@$ParamsTab, $Param0);
		return %RES;
	}
}

add_get '/list' => sub { 
	my $fmt = shift; my %MEM = vr::memoriesList();
	if (config->{'filter-parameters'}) {
		%MEM = xmlFlatten (\%MEM, [ @{config->{'filter-parameters'}} ]);
    } else {
		%MEM = ( " parameters='none' " => $MEM{__DEFAULT__} );    # compatibility
	}
	
	content_type $fmt->MIME();
	return $fmt->envelope (['OK'], $fmt->listMemories(%MEM));	
};

#	------------------ OmegaT --------------------------------

get '/omegat' => sub {
	my $config = config->{omegat}; my %spec = vr::fillDatabaseParams(); my $item = vr::findMemory(\%spec, param('inherit'));
	unless ($item) {
		my $Params = ""; while (my ($k, $v) = each (%{params('query')})) { $Params .= "\t<param name='$k' value='$v' />\n" unless ref $v; }
		return << "EOF";
<?xml version='1.0' encoding='UTF-8' ?>
<response server='$main::SERVER' status='error' cause='No memory found with such parameters'>
$Params
</response>
EOF
	}
    my $tableInfo = "table=$item->{table_schema}.$item->{table_name}\n";
    if (param('inherit')) {
        my $parent = $item;
		if ($item->{real_parent} || $item->{realparent}) {	# use meta_info if available
			my $level = param('inherit') - 1; $parent = $item->{real_parent} || $item->{realparent};
			until ($level == 0) {
				my $sql = "select * from " . (ref(config->{list}{sql}{view}) ? config->{list}{sql}{view}{name} : config->{list}{sql}{view});
				if ($parent =~ /^\d+$/) {
					if ($item->{mem_id}) { $sql .= " where mem_id = $parent"; ($parent) = $item->{DBI}->selectrow_href ($sql); $parent = $parent->{mem_id}; }
					if ($item->{mem_code}) { $sql .= " where mem_code = $parent"; ($parent) = $item->{DBI}->selectrow_href ($sql); $parent = $parent->{mem_code}; }					
				} elsif ($parent =~ m!/! or $parent =~ m!:!) {
					$sql .= " where mem_path = '$parent'"; ($parent) = $item->{DBI}->selectrow_href ($sql); $parent = $parent->{mem_path};
				} else {
					my ($parentSchema,$parentName) = split(/\./, $parent); ($parentSchema,$parentName) = ('public',$parentSchema) unless $parentName;
					$sql .= sprintf(" where table_schema='%s' and table_name='%s'", $parentSchema, $parentName);
					($parent) = $item->{DBI}->selectrow_href ($sql); $parent = "$parent->{table_schema}.$parent->{table_name}";
				}				
			}
		} else {	# if meta_info is not available, then function 'parent_table' must be defined
			$parent = "$item->{table_schema}.$item->{table_name}";
			for (my $level = param('inherit'); $level > 0; $level--) {
				($parent) = $item->{DBI}->selectrow_array ("select parent_table('$parent')");
			}
		}		
        $tableInfo = "table.write=$item->{table_schema}.$item->{table_name}\n";
        $tableInfo .= "table.read=$parent\n";
		$item->{DBI}->disconnect;
    }
	
	my $PROPS = '';
	while (my ($k, $v) = each (%{config->{'omegat'}{'props-display'}})) { $PROPS .= "props.display.$k=$v\n"; }

	my $HEAD;
	if (my $proxy = config->{'http-proxy'}) {
		$HEAD = "class=org.silvestrislab.cyclotis.omegat.tm.HttpMemory\n";
		$HEAD .= "url=$proxy?";
		if (param('inherit')) { 
			$HEAD .= "read=$1&" if $tableInfo =~ /table\.read=(.+?)\n/;
			$HEAD .= "write=$1&" if $tableInfo =~ /table\.write=(.+?)\n/;			
		} 
		else { $HEAD .= $tableInfo . "\n"; }
	} else {
		$HEAD = << "EOF";
class=org.silvestrislab.cyclotis.omegat.tm.PostgresqlMemory

jdbc.host=$item->{host}
jdbc.port=$item->{port}
jdbc.db=$item->{db}

jdbc.user=tm
jdbc.password=tm

$tableInfo
EOF
	}
	
	header 'Content-Type' => 'text/plain';
	header 'Content-Disposition' => 'attachment;filename=' . param('attach') if param('attach');
	
	return << "EOF";
$HEAD
update=true

log.file=Silvestris.log
log.list=update,search,javascript,sql

provider.name=$config->{'provider-name'}

$PROPS

EOF
};

# 	----------------- Parameters for Trados -----------------

=head2 GET /trados-params

Shows parameters for Trados

=cut

get '/trados-params' => sub {
	my $dbParams = config->{trados}{uri} || 'directdb://$host:$port/?db=$catalog';  $dbParams =~ s/\$(\w+)/param($1)/eg;
	return "<response server='$main::SERVER' status='OK'>$dbParams</response>";
};


# 	------------------ Default route -------------------------

any qr{.*} => sub {
	my $fmt = "formats::$1" if request->uri =~ /\.(\w+)(\?|$)/; $fmt ||= "formats::" . config->{formats}{default};

	my ($CMD) = (request->uri =~ m!^/(\w+)!);
	if ($CMD) { $CMD = "Bad command: $CMD"; } else { $CMD = 'no command'; }
	return $fmt->message(warn => $CMD);
};

dance;

=head1 License

Copyright 2014-2016 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
