/*
 * Decompiled with CFR 0.152.
 */
package org.omegat.core.spellchecker;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.IOUtils;
import org.languagetool.JLanguageTool;
import org.omegat.core.Core;
import org.omegat.core.CoreEvents;
import org.omegat.core.data.SourceTextEntry;
import org.omegat.core.events.IEntryEventListener;
import org.omegat.core.events.IProjectEventListener;
import org.omegat.core.spellchecker.ISpellChecker;
import org.omegat.core.spellchecker.ISpellCheckerProvider;
import org.omegat.core.spellchecker.SpellCheckerDummy;
import org.omegat.core.spellchecker.SpellCheckerJMySpell;
import org.omegat.core.spellchecker.SpellCheckerLangToolHunspell;
import org.omegat.tokenizer.ITokenizer;
import org.omegat.util.Language;
import org.omegat.util.Log;
import org.omegat.util.Preferences;
import org.omegat.util.StaticUtils;
import org.omegat.util.Token;

public class SpellChecker
implements ISpellChecker {
    public static final File DEFAULT_DICTIONARY_DIR = new File(StaticUtils.getConfigDir(), "spelling");
    private ISpellCheckerProvider checker;
    private List<String> ignoreList = new ArrayList<String>();
    private List<String> learnedList = new ArrayList<String>();
    private final Set<String> correctWordsCache = new HashSet<String>();
    private final Set<String> incorrectWordsCache = new HashSet<String>();
    private Path ignoreFilePath;
    private Path learnedFilePath;

    public SpellChecker() {
        CoreEvents.registerProjectChangeListener(new IProjectEventListener(){

            @Override
            public void onProjectChanged(IProjectEventListener.PROJECT_CHANGE_TYPE eventType) {
                switch (eventType) {
                    case LOAD: 
                    case CREATE: {
                        SpellChecker.this.initialize();
                        break;
                    }
                    case CLOSE: {
                        SpellChecker.this.destroy();
                        break;
                    }
                }
                SpellChecker.this.resetCache();
            }
        });
        CoreEvents.registerEntryEventListener(new IEntryEventListener(){

            @Override
            public void onNewFile(String activeFileName) {
                SpellChecker.this.resetCache();
            }

            @Override
            public void onEntryActivated(SourceTextEntry newEntry) {
            }
        });
    }

    @Override
    public void initialize() {
        Language targetLanguage = Core.getProject().getProjectProperties().getTargetLanguage();
        Stream<String> toCheck = Stream.of(targetLanguage.getLocaleCode(), targetLanguage.getLocaleCode().replace('_', '-'), targetLanguage.getLanguageCode());
        this.checker = (ISpellCheckerProvider)toCheck.map(SpellChecker::initializeWithLanguage).filter(Optional::isPresent).findFirst().orElseGet(() -> Optional.of(new SpellCheckerDummy())).get();
        if (this.checker instanceof SpellCheckerDummy) {
            Log.log("No spell checker found for language " + targetLanguage);
        }
        this.loadWordLists();
    }

    private static Optional<ISpellCheckerProvider> initializeWithLanguage(String language) {
        String dictionaryDir = Preferences.getPreferenceDefault("spellcheker_dir", DEFAULT_DICTIONARY_DIR.getPath());
        File dictBasename = new File(dictionaryDir, language);
        File affixName = new File(dictionaryDir, language + ".aff");
        File dictionaryName = new File(dictionaryDir, language + ".dic");
        if (!dictionaryName.exists()) {
            SpellChecker.installBundledDictionary(dictionaryDir, language);
        }
        if (!dictionaryName.exists()) {
            SpellChecker.installLTBundledDictionary(dictionaryDir, language);
        }
        if (!SpellChecker.isValidFile(affixName) || !SpellChecker.isValidFile(dictionaryName)) {
            return Optional.empty();
        }
        try {
            SpellCheckerLangToolHunspell result = new SpellCheckerLangToolHunspell(dictBasename.getPath());
            Log.log("Initialized LanguageTool Hunspell spell checker for language '" + language + "' dictionary " + dictionaryName);
            return Optional.of(result);
        }
        catch (Throwable ex) {
            Log.log("Error loading hunspell: " + ex.getMessage());
            try {
                SpellCheckerJMySpell result = new SpellCheckerJMySpell(dictionaryName.getPath(), affixName.getPath());
                Log.log("Initialized JMySpell spell checker for language '" + language + "' dictionary " + dictionaryName);
                return Optional.of(result);
            }
            catch (Exception ex2) {
                Log.log("Error loading jmyspell: " + ex2.getMessage());
                return Optional.empty();
            }
        }
    }

    private static boolean isValidFile(File file) {
        try {
            if (!file.exists()) {
                return false;
            }
            if (!file.isFile()) {
                Log.log("Spelling dictionary exists but is not a file: " + file.getPath());
                return false;
            }
            if (!file.canRead()) {
                Log.log("Can't read spelling dictionary: " + file.getPath());
                return false;
            }
            if (file.length() == 0L) {
                Log.log("Spelling dictionary appears to be empty: " + file.getPath());
                return false;
            }
            return true;
        }
        catch (Throwable ex) {
            Log.log(ex);
            return false;
        }
    }

    private static void installBundledDictionary(String dictionaryDir, String language) {
        try (InputStream bundledDict = SpellChecker.class.getResourceAsStream(language + ".zip");){
            if (bundledDict == null) {
                return;
            }
            StaticUtils.extractFromZip(bundledDict, new File(dictionaryDir), Arrays.asList(language + ".aff", language + ".dic")::contains);
        }
        catch (IOException e) {
            Log.log(e);
        }
    }

    private static void installLTBundledDictionary(String dictionaryDir, String language) {
        String resPath = "/" + new Language(language).getLanguageCode() + "/hunspell/" + language + ".dic";
        if (!JLanguageTool.getDataBroker().resourceExists(resPath)) {
            return;
        }
        try {
            FileOutputStream fos;
            try (InputStream dicStream = JLanguageTool.getDataBroker().getFromResourceDirAsStream(resPath);){
                fos = new FileOutputStream(new File(dictionaryDir, language + ".dic"));
                try {
                    IOUtils.copy(dicStream, (OutputStream)fos);
                }
                finally {
                    fos.close();
                }
            }
            try (InputStream affStream = JLanguageTool.getDataBroker().getFromResourceDirAsStream(resPath.replaceFirst(".dic$", ".aff"));){
                fos = new FileOutputStream(new File(dictionaryDir, language + ".aff"));
                try {
                    IOUtils.copy(affStream, (OutputStream)fos);
                }
                finally {
                    fos.close();
                }
            }
        }
        catch (Exception ex) {
            Log.log(ex);
        }
    }

    private void loadWordLists() {
        String projectDir = Core.getProject().getProjectProperties().getProjectInternal();
        this.ignoreFilePath = Paths.get(projectDir, "ignored_words.txt");
        this.ignoreList.clear();
        if (this.ignoreFilePath.toFile().isFile()) {
            try {
                this.ignoreList.addAll(Files.readAllLines(this.ignoreFilePath, StandardCharsets.UTF_8));
            }
            catch (Exception ex) {
                Log.log(ex);
            }
        }
        this.learnedFilePath = Paths.get(projectDir, "learned_words.txt");
        this.learnedList.clear();
        if (this.learnedFilePath.toFile().isFile()) {
            try {
                this.learnedList.addAll(Files.readAllLines(this.learnedFilePath, StandardCharsets.UTF_8));
                this.learnedList.stream().forEach(word -> this.checker.learnWord((String)word));
            }
            catch (Exception ex) {
                Log.log(ex);
            }
        }
    }

    @Override
    public void destroy() {
        this.saveWordLists();
        if (this.checker != null) {
            this.checker.destroy();
            this.checker = null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void resetCache() {
        SpellChecker spellChecker = this;
        synchronized (spellChecker) {
            this.incorrectWordsCache.clear();
            this.correctWordsCache.clear();
        }
    }

    @Override
    public void saveWordLists() {
        try {
            Files.write(this.ignoreFilePath, this.ignoreList, new OpenOption[0]);
        }
        catch (IOException ex) {
            Log.log(ex);
        }
        try {
            Files.write(this.learnedFilePath, this.learnedList, new OpenOption[0]);
        }
        catch (IOException ex) {
            Log.log(ex);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean isCorrect(String word) {
        if (this.checker == null) {
            return true;
        }
        word = SpellChecker.normalize(word);
        SpellChecker spellChecker = this;
        synchronized (spellChecker) {
            if (this.incorrectWordsCache.contains(word)) {
                return false;
            }
            if (this.correctWordsCache.contains(word)) {
                return true;
            }
        }
        boolean isCorrect = this.learnedList.contains(word) || this.ignoreList.contains(word) ? true : this.checker.isCorrect(word);
        SpellChecker spellChecker2 = this;
        synchronized (spellChecker2) {
            if (isCorrect) {
                this.correctWordsCache.add(word);
            } else {
                this.incorrectWordsCache.add(word);
            }
        }
        return isCorrect;
    }

    @Override
    public List<String> suggest(String word) {
        if (this.isCorrect(word)) {
            return Collections.emptyList();
        }
        return this.checker.suggest(SpellChecker.normalize(word));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void ignoreWord(String word) {
        if (!this.ignoreList.contains(word = SpellChecker.normalize(word))) {
            this.ignoreList.add(word);
            SpellChecker spellChecker = this;
            synchronized (spellChecker) {
                this.incorrectWordsCache.remove(word);
                this.correctWordsCache.add(word);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void learnWord(String word) {
        if (!this.learnedList.contains(word = SpellChecker.normalize(word))) {
            this.learnedList.add(word);
            this.checker.learnWord(word);
            SpellChecker spellChecker = this;
            synchronized (spellChecker) {
                this.incorrectWordsCache.remove(word);
                this.correctWordsCache.add(word);
            }
        }
    }

    @Override
    public boolean isIgnoredWord(String word) {
        return this.ignoreList.contains(SpellChecker.normalize(word));
    }

    @Override
    public boolean isLearnedWord(String word) {
        return this.learnedList.contains(SpellChecker.normalize(word));
    }

    private static String normalize(String word) {
        return word.replace('\u2019', '\'');
    }

    @Override
    public List<Token> getMisspelledTokens(String text) {
        return Stream.of(Core.getProject().getTargetTokenizer().tokenizeWords(text, ITokenizer.StemmingMode.NONE)).filter(tok -> !this.isCorrect(tok.getTextFromString(text))).collect(Collectors.toList());
    }
}

