package DbCreator;

sub exist_cmd($) {
    my $cmd = shift; foreach (split(/:/, $ENV{PATH})) { return 1 if -x "$_/$cmd"; } 
    return 0;
}

# Create minimal database (no tables, but pg_trgm loaded)
sub createDatabase {
   local $ENV{PGDATESTYLE} = 'ISO';	# REST client is based on this format
   my $dbParam = $_[-1] || die "Missing database name\nSyntax: $0 $command <database name> <description>?"; $dbParam = $_[-2] if $dbParam =~ /\s/; # Description
   if ($dbParam =~ /[\@\:]/) {
		my ($name, $host, $port, $user, $pass, $owner) = &analyse_dbSpec($dbParam);
		if (exist_cmd('createdb')) {
			unshift(@_, '-h' => $host) if $host and ($host ne 'localhost') and ($host ne '127.0.0.1');
			unshift(@_, '-p' => $port) if $port =~ /^\d+$/;
			unshift(@_, '-U' => $user) if $user;
			unshift(@_, '-O' => $owner) if $owner;
			if ($_[-1] =~ /\s/) { $_[-2] = $name; } else { $_[-1] = $name; }
			# Use the Postgresql command-line clients, not DBI.
			system 'createdb', '-w', @_;
        } elsif (require DBI) {
            # Use all parameters except name
            my $conn = "DBI:Pg:"; 
            $conn .= "host=" . ($host || 'localhost'); # here host is mandatory
            $conn .= ";port=$port" if $port; # optional			
			$conn = DBI->connect($conn, $user, $pass) or die "Could not connect to Postgres. Please control that you are allowed to connect like this";
			$conn->do("create database $name " . ($owner ? " OWNER $owner" : ""));
			if ($conn->{pg_server_version} >= 90100) { $conn->do("create extension pg_trgm"); $conn->disconnect; }
			else { $conn->disconnect; createExtension($dbParam, 'pg_trgm', 8); return; }
        } else {
            die "No way to create database: either install DBI or command-line tools!";
        }
   } else { # Only the name of the database
		if (exist_cmd('createdb')) { system 'createdb', '-w', $dbParam; } 
        elsif (require DBI) {
            # Use all parameters except name
			my $conn = DBI->connect("DBI:Pg:host=localhost") or die "Could not connect to Postgres. Please control that you are allowed to connect like this";
			$conn->do("create database $dbParam");
			if ($conn->{pg_server_version} >= 90100) { $conn->do("create extension pg_trgm"); $conn->disconnect; }
			else { $conn->disconnect; createExtension($dbParam, 'pg_trgm', 8); return; }
        } else {
            die "No way to create database: either install DBI or command-line tools!";
        }        
   }
   
   # Mandatory : load pg_trgm
   createExtension($dbParam, 'pg_trgm');   
}

sub initDatabase {
   my $dbParam = $_[-1] || die "Missing database name\nSyntax: $0 init-db <database name>"; 

   # Mandatory : load pg_trgm
   createExtension($dbParam, 'pg_trgm');   
}

#  ----------- commands -------------

sub doCreate_schema {
	my $dbh = shift; my $schema = shift;
	unless ($dbh->do('create schema ' . $schema)) { 
		if ($dbh->state eq '42P06') { print STDERR "Schema $schema already exists\n"; } else { die $DBI::errstr; }
	}
	foreach (@_) {
		if (/^owner=(.+)$/i) {
			unless ($dbh->do('alter schema ' . $schema . qq( owner to "$1"))) { 
				print STDERR "Could not change schema owner: $DBI::errstr";
			}		
		}
		if (/^(users|writers)=(.+)$/i) {
			foreach my $user (split(/,/, $2)) {
				DbCreator::grant ('create', $user, "schema $schema", $dbh);
			}
		}
	}
   print STDERR "Schema $schema created OK\n"; 
}

sub doCreate_rule {
   my $dbh = shift;
   my $SYNTAX = "$0 create-rule <db name> <table name> <key field> <action> <column=action>+ (see manual or help for details)";
   my $table = shift || die "$0 create-rule: Missing table name\nSyntax: $SYNTAX"; 
   my $keyField = shift || die "$0 create-rule: Missing key field\nSyntax: $SYNTAX"; die "Unknown key field : $keyField" unless $keyField =~ /^au(th(or)?)?|c(on)?te?xt|s(ou)?rc/;
   my $action = shift || 'replace-only'; die "Unknown action : $action" unless $action =~ /(replace-(only|null|delete))|(version)/;
   my $keyFieldSql = 'SRC = NEW.SRC';
   $keyFieldSql .= ' and AUTHOR=NEW.AUTHOR' if $keyField =~ /au(th(or)?)?/;
   $keyFieldSql .= ' and CONTEXT=NEW.CONTEXT' if $keyField =~ /c(on)?te?xt/;
   $keyFieldSql = substr($keyFieldSql, 18) if $keyField =~ /c(on)?te?xt-only/;	# context is unique, no source
   my $parent = $table; if ($table =~ /^(.+):(.+)$/) { ($table, $parent) = ($1, $2); }
   my $setChangerAndDate = "UPDATE $parent 
	SET CHANGER=COALESCE(NEW.CHANGER,NEW.AUTHOR), CHANGEDATE=COALESCE(NEW.CHANGEDATE,NEW.DATE,NOW())";
   my ($tableSchema, $tableName) = split(/\./, $table); unless ($tableName) { $tableName = $tableSchema; $tableSchema = 'public'; }
   my ($existChanger) = $dbh->selectrow_array (
		"select column_name from information_schema.columns
		  where table_schema = '$tableSchema' and table_name = '$tableName' and column_name like 'ch%'"
   );
   unless ($existChanger) {
		$setChangerAndDate =~ s/CHANGER/AUTHOR/g; $setChangerAndDate =~ s/CHANGEDATE/DATE/g;
		$setChangerAndDate =~ s/COALESCE\((.+),\1\)/$1/g;
   }
   
   # Rule for insertion
   my $table_changename = $table; $table_changename =~ s/\./\_\_/;
   my $insertSql = "CREATE RULE INS_$table_changename AS
      ON INSERT TO $table WHERE EXISTS(SELECT * FROM $parent WHERE $keyFieldSql)";
   if ($action =~ /version/) {
       # Update the old one segment's changedate/author, but insert the new one as well
       $insertSql .= " DO ALSO $setChangerAndDate  WHERE $keyFieldSql and CHANGEDATE is null and TRA != NEW.TRA";
   } else {
	   my $changeContents = 'TRA=NEW.TRA';
	   # What to do with props and note: default is to replace with new value
	   foreach my $colName ('props', 'note') {
			my ($exist) = $dbh->selectrow_array (
				"select column_name from information_schema.columns
				  where table_schema = '$tableSchema' and table_name = '$tableName' and column_name = '$colName'"
			);
			next unless $exist;
			my $colAction = 'replace'; # default
			foreach my $arg (@_) { $colAction = lc($1) if $arg =~ /^$colName=(.+)$/; }
			if ($colAction eq 'replace') { $changeContents .= ", $colName = NEW.$colName"; }
			elsif ($colAction eq 'concat') { $changeContents .= ", $colName = concat($colName, NEW.$colName)"; } # TEXT's concat
			elsif ($colAction eq 'merge') { $changeContents .= ", $colName = $colName || NEW.$colName"; } # Supposed to be an HSTORE
			elsif ($colAction eq 'keep') { } # do nothing
			else { die "Unknown action for column $colName : $colAction"; }
	   }
       # Update the old one segment's changedate/author and contents. The old one is lost
       $insertSql .= " DO INSTEAD $setChangerAndDate, $changeContents WHERE $keyFieldSql and TRA != NEW.TRA";
   }
   $dbh->prepare($insertSql)->execute or die $DBI::errstr;
   print STDERR "Create insert rule for $action on $table ($keyField) OK\n";
   
   # Rule for deletion
   my $deleteSql = "CREATE RULE DEL_$table_changename AS
      ON DELETE TO $table
      DO INSTEAD ";
   $keyFieldSql =~ s/NEW/OLD/g; 
   if ($action =~ /version/) {
       # No physical deletion, simply set an expiration date instead
       $deleteSql .= "UPDATE $parent set CHANGEDATE=NOW() WHERE $keyFieldSql and CHANGEDATE is null";
       $dbh->prepare($deleteSql)->execute or die $DBI::errstr;
       print STDERR "Create delete rule for $action on $table ($keyField) OK\n";	   
   } elsif ($action =~ /replace-(null|delete)/) {
       # No physical deletion, we replace it by set tra = null and update changedate/author
       $deleteSql .= "UPDATE $parent set CHANGEDATE=NOW(), TRA=null WHERE $keyFieldSql";
       $dbh->prepare($deleteSql)->execute or die $DBI::errstr;	   
       print STDERR "Create delete rule for $action on $table ($keyField) OK\n";
   }
   # else : action = replace-only, no deletion   
}

our %reqStr;
sub doCreate_function {
	unless ($reqStr{FUNCTION}) {
		if (open(SQL, '< functions.sql')) {
		   local $/ = 'language sql;';
		   while (<SQL>) {
			  $reqStr{uc($1)}{lc($2)} = $_ if /CREATE(?:\s+OR\s+REPLACE)?\s+(\w+)\s+(\w+)/i;
		   }
		   close (SQL);
		}
	}

   my $dbh = shift;
	my $sql = $reqStr{FUNCTION}{lc(shift)};
	our @lib; push (@lib, $1) while $sql =~ s/--\s+using\s+(\w+)\s*\n//g; 
	$sql =~  s/--.+\n//g;  my $st = $dbh->prepare($sql);
	unless ($st->execute) {
          &createExtension ($dbname, $_) foreach @lib;
          $st->execute or die $DBI::errstr;
	}
	$dbh->disconnect; exit;
}

sub doCreate_index {
   my $dbh = shift;
   my $table = shift || die "$0 create-index : Missing table name\nSyntax: $0 create-index <db name> <table name> <index name>? <language>?";
   my $idxName = shift || 'fuzzy'; my $lang = shift || 'simple';
   unless ($lang) {	# try to read language from meta_info table
		my $searchLang = $dbh->prepare('select src_lang,tra_lang from public.meta_info where table_name=? and table_schema=?');
		my ($shema,$table1) = ('public',$table); ($schema,$table1) = ($1,$2) if $table =~ /^(.+)\.(.+)$/;
		if ($searchLang->execute ($schema,$table1)) {
			my @res = $searchLang->fetchrow_array;
			$lang ||= $res[2] if $idxName =~ /Tra/i; $lang ||= $res[1];
		}
   }   
   
   my $table_changename = $table; $table_changename =~ s/\./\_\_/;
   my $sql = "create index idx_${table_changename}_${idxName} ON $table ";
   if ($idxName =~ /all|date/i) {
      $sql .= "(CHANGEDATE)";
   }
   if ($idxName =~ /(conc|glos)Src/i) {
      $sql .= "USING gist(to_tsvector('$lang', src))";
   }
   if ($idxName =~ /(conc|glos)Tra/i) {
      $sql .= "USING gist(to_tsvector('$lang', tra))";
   }
   if ($idxName =~ /fuzzy/i) {
      $sql .= "USING gist(src gist_trgm_ops)";
   }
   die "Unknown index : $idxName" unless $sql =~ /\)\s*$/;
   if (my $ops = shift) {
      my $tags = shift || '<.+?>';
      $sql =~ s/(src|tra)/regexp_replace($&,E'$tags'::text, ''::text,'g')/g if $ops =~ /untag/;
      $sql =~ s/(src|tra)/tokenize($&,'${lang}_stem',true)/g and $sql =~ s/simple_stem/simple/ if $ops =~ /tokenize/;
   }
   print STDERR $sql, "\n";
   my $st = $dbh->prepare($sql);
   unless ($st->execute) {
          &DbCreator::createExtension ($dbParam, 'btree_gist');
          $st->execute or die $DBI::errstr;
   }
}

sub doCreate_dictionary {
   my $dbh = shift;
	my $lang = shift || die "Syntaxe: $0 create-dict [database] [langName] [dict-file]? [affix-file]? [stop-words]?";
	my $dictFile = shift || 'ispell_sample'; my $affix = shift || 'ispell_sample'; my $stop = shift || $lang;
   my $sql = "CREATE TEXT SEARCH DICTIONARY ispell_$lang(TEMPLATE = ispell, DictFile = $dictFile, AffFile = $affix, StopWords = $stop)";
   my $st = $dbh->prepare($sql);
   $st->execute or die $DBI::errstr;
   $dbh->disconnect; exit;	
}

sub doCreate_textSearchConfig {
   my $dbh = shift;
	my $lang = shift || die "Syntaxe: $0 create-text-search-config [database] [langName] [origin] [token=dictionnaries]+";
	my $origin = shift || die "Syntaxe: $0 create-text-search-config [database] [langName] [origin] [token=dictionnaries]+";
   $dbh->do("CREATE TEXT SEARCH CONFIGURATION text_$langName ( Copy = $origin )") or die $DBI::errstr;
   my %tokens; foreach (@ARGV) { $tokens{$1} = $2 if /(\w+)=(.*)$/; }
   while (my ($k, $v) = each (%tokens)) {
		if ($v) {
			$dbh->do("ALTER TEXT SEARCH CONFIGURATION text_$langName ALTER MAPPING FOR $k WITH $v") or die $DBI::errstr;		
		} else {
			$dbh->do("ALTER TEXT SEARCH CONFIGURATION text_$langName DROP MAPPING FOR $k") or die $DBI::errstr;		
		}
   }
   $dbh->disconnect; exit;	
}

#  ----------- Tools -------------

# Loads an extension - if PostgreSQL >= 9.1, use 'create extension', else use share directory
sub createExtension {
	my $dbSpec = shift; # in our local format, may contain host:port, etc.
	my $extensionName = shift;
	
	print STDERR "Loading extension $extensionName into $dbSpec\n";
	# Warning: pg_config does not use port or host: be sure to configure the PATH to the correct version!!!
	my $version = shift || `pg_config --version`; $version = $& if $version =~ /(\d+)\.(\d+)/;
	if ($version < 9.1) { 
		my $shareDir = `pg_config --sharedir`; $shareDir =~ s/\r?\n?$//;
		my ($name, $host, $port) = &analyse_dbSpec($dbSpec);
		$dbSpec = "-d $name"; $dbSpec .= " -h $host" if $host; $dbSpec .= " -p $port" if $port;
		system "psql $dbSpec < $shareDir/contrib/$extensionName.sql";
	} else {
		require DBI or return createExtension($dbSpec, $extensionName, 8);
 		
		my $dbiString = &analyse_dbSpec($dbSpec); # scalar : DBI string
		my ($user, $pass);
		if ($dbSpec =~ s/username=(.+?)(;|$)// or $dbiString =~ s/username=(.+?)(;|$)//) { $user = $1; }
		if ($dbSpec =~ s/password=(.+?)(;|$)// or $dbiString =~ s/password=(.+?)(;|$)//) { $pass = $1; }
		$dbiString =~ s/owner=(.+?)(;|$)//; # non-used parameter
		my $dbh = DBI->connect($dbiString,$user,$pass) or return createExtension($dbSpec, $extensionName, 8);
		$dbh->prepare ('create extension ' . $extensionName)->execute or die $DBI::errstr;
		$dbh->disconnect;
	}
}

=head2 Database name

By default the name you type is considered as the catalog name, and we suppose that server is on localhost, port 5432, connecting as current user without password.

If it is no true, you can use the following format:

=over 1

=item *

a host, separated by '/' (for example, h/base means 'dbname = base', host = 'h')

=item *

a port, separated by ':' 

For example:

':5433/base' means 'dbname = 'base', port=5433; 

'h:5433/base' means 'dbname = base', host = 'h', port=5433

=item *

a user, separated by @

For example

me@base means 'user = me, dbname = base'

me@h/base means 'user = me, dbname = base, host = h'

=item *

user password, separated from user by :

me:123@base means 'user = me, password = 123, dbname = base'

=item *

owner of object (database catalog, schema, table) after creation

me:123#you@base means 'user = me, password = 123, owner = you dbname = base'

me#you@base means 'user = me, owner = you, dbname = base'

(note: in this case, user and password are used for connection; you do not need to know owner's password, if you have creation rights)

=back

=cut
# Convert Cyclotis database specification to (dbname, host, port, $user, $pass, $dbi_string)
sub analyse_dbSpec {
	my $dbnameSpec = shift; my ($dbName, $host, $port, $user, $pass, $owner) = ($dbnameSpec,undef,undef, undef, undef, undef);
	# Hypothesis 1: specified by command line Cyclotis format
	if ($dbnameSpec =~ /^(.+)\@(.+)$/) { 
		($user, $dbName) = ($1,$2); 
		($user, $owner) = split(/\#/, $user); ($user, $pass) = split(/:/, $user);  
	}
	else { $dbName = $dbnameSpec; } 
	if ($dbName =~ m!^(.*)/(.+)$!) { ($host, $dbName) = ($1,$2); } # else $dbName = $dbnameSpec; 
	if ($host) { ($host,$port) = split(/\:/, $host); }
	# Hypothesis 2: specified as a DBI string
	$dbName = $2 if $dbnameSpec =~ /db(name)?=(.+?)(;|$)/;
	$host = $1 if $dbnameSpec =~ /host=(.+?)(;|$)/;
	$port = $1 if $dbnameSpec =~ /port=(.+?)(;|$)/;
	
	$dbnameSpec = "dbi:Pg:dbname=$dbName";
	$dbnameSpec .= ";host=$host" if $host and $host ne 'localhost' and $host ne '127.0.0.1';
	$dbnameSpec .= ";port=$port" if $port =~ /^\d{1,5}$/;;
	$dbnameSpec .= ";username=$user" if $user;
	$dbnameSpec .= ";password=$pass" if $pass;
	$dbnameSpec .= ";owner=$owner" if $owner;
	
	if (wantarray) { return ($dbName, $host, $port, $user, $pass, $owner, $dbnameSpec); }
	else { return $dbnameSpec; } 
}


=head2 DbCreator::grant ('priv1,priv2,priv3', $user, $object, $dbh)

Grant set of privileges to $user for $object

$user can be :

=over 1

=item *

A single user name 

=item *

A group name preceeded by ':'

=item *

The string '*' for all users

=back

=cut
sub grant {
	my ($privileges, $user, $object, $dbh) = @_;
	my $st = "grant $privileges on $object";
	if ($user eq '*') { $st .= ' to public'; }
	elsif ($user =~ s/^://) { $st .= qq( to group "$user"); }
	else { $st .= qq( to "$user"); }
	$dbh->do($st);
}

1;

=head1 License

Copyright 2013-2015 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
