=head1 DESCRIPTION

A simple way to build the SQL search query

=head1 USAGE

	my $search = new Silvestris::Cyclotis::Database::Query::Search(
		schema => $schema, table => $table,
		cmd => 'fuzzy'
	);
	# Add parameters - each call to such a method creates a new query with additionnal parameters (original query is not modified)
	$search = $search->addParam_version('2013-11-20 11:23:23');
	$search = $search->addParam_uniq ('field1', 'field2');
	# Use the built query
	my $sql = $search->toSql;
	my $statement = $search->execute ($dbi::connection, $query_text);

=cut
package Silvestris::Cyclotis::Database::Query::Search;
		
=head1 METHODS
	
=head2 Silvestris::Cyclotis::Database::Query::Search->new (initial parameters)

Build The initial query

=head3 Initial parameters

=over 1

=line-item tableRef:	Instance of Silvestris::Cyclotis::Database::Table
=line-item tableName:	The database table to connect. May contain a schema prefix or not.
=line-item schema:		The Postgresql schema of the table. Defaults to public
=line-item cmd:			The Cyclotis search mode. The list of search modes will be displayed later in this document.

=back

=cut
sub new {
	my ($class, %params) = @_; our %CMD;
	my $table = $params{tableName}; $table = "$params{schema}.$table" if $params{schema};
	my %obj = (tableName => $table, tableRef => $params{tableRef}, cmd => $params{cmd}, sqlConfig => $params{sqlConfig}, tags => $params{tags});
	$obj{displayScore} = 'similarity(src,:txt)'; $obj{select} = [ '*', 'similarity(src,:txt) as score' ];
	my @fields = (); @fields = @{$params{fields}} if ref $params{fields};
	my $cmdSql = $class->sqlExpressionForCommand($params{cmd}, $params{tableRef});
	if ($params{tableRef}{mem_id} and $params{tableRef}->standard_parent()) {	# mode by_id
		$obj{from} = [ $obj{std_parent} = $params{tableRef}->standard_parent() ]; 
		$obj{mem_id} = $obj{bind}{':mem_id'} = $params{tableRef}{mem_id};
		if (ref($params{tableRef}{children}) and (scalar(@{$params{tableRef}{children}}) > 0)) { 
			$obj{where} = [ "mem_id = any (:childListTxt::int[])", 'and', "($cmdSql)" ]; 
			$obj{bind}{':childListTxt'} = sprintf('{%s}', join(',', $obj{mem_id}, @{$params{tableRef}{children}}));
		} 
		else { $obj{where} = [ 'mem_id=:mem_id', 'and', "($cmdSql)" ]; $obj{mem_id} = $obj{bind}{':mem_id'} = $params{tableRef}{mem_id}; }
		_addMetaFields(\%obj, \@fields, 'mem_id');
	} elsif ($params{tableRef}{mem_code} and $params{tableRef}->standard_parent()) {	# mode by_code
		$obj{from} = [ $obj{std_parent} = $params{tableRef}->standard_parent() ]; 
		$obj{mem_code} = $obj{bind}{':mem_code'} = $params{tableRef}{mem_code};
		if (defined($params{tableRef}{max_code})) { # meta view must have a field containing max value
			$obj{where} = [ "mem_code >= :mem_code", 'and', 'mem_code < :max_code', 'and', "($cmdSql)" ]; 
			$obj{bind}{":max_code"} = 1 + ($obj{max_code} = $params{tableRef}{max_code});
		}
		else { $obj{where} = [ 'mem_code=:mem_code', 'and', "($cmdSql)" ];  }
		_addMetaFields(\%obj, \@fields, 'mem_code');
	} elsif ($params{tableRef}{mem_path}) {		# mode by_path
		my $separator = $1 if $params{tableRef}{mem_path} =~ m!([:/])!;		
		$obj{from} = [ 'public.' . substr($params{tableRef}{mem_path}, 0, index($params{tableRef}{mem_path},$separator)) ];
		$obj{where} = [ 'mem_path >= :mem_path1', 'and', 'mem_path < :mem_path2', 'and', "($cmdSql)" ];
		$obj{mem_path} = $obj{bind}{':mem_path1'} = $params{tableRef}{mem_path};
		$obj{bind}{':mem_path2'} = $params{tableRef}{mem_path}; $obj{bind}{':mem_path2'} =~ s/(.)$/chr(ord($1) + 1)/e;			
		if ($separator eq ':') { 	# by path, no lang
			unless (_addMetaFields(\%obj, \@fields, 'mem_path')) {
				push (@{$obj{select}}, "substring(mem_path from E':(\\w+\\.\\w+)\\Z') as mem_name") if grep { /mem_name/i } @fields; # no jointure needed, as mem_name is a part of mem_path			
			}
		} elsif ($separator eq '/') {	# by path, with langs
			push (@{$obj{select}}, "substring(mem_path from E'/(\\w+\\.\\w+)(\([\\w\\-,]+\\))?\\Z') as mem_name") if grep { /mem_name/i } @fields; # no jointure needed, as mem_name is a part of mem_path
			push (@{$obj{select}}, "substring(mem_path from E'\(([\\w\\-]+),') as src_lang") if grep { /src_lang/i } @fields; # no jointure needed, as source lang is a part of mem_path
			push (@{$obj{select}}, "substring(mem_path from E',([\\w\\-]+)\\)') as tra_lang") if grep { /tra_lang/i } @fields; # no jointure needed, as target lang is a part of mem_path
		}
	} else {	# mode by_void, or could not find std_parent
		$obj{from} = [ $table ];
		$obj{where} = [ "($cmdSql)" ];
	}
	if ($params{cmd} =~ /Tra/) {
		s/\bsrc\b/tra/i foreach @{$obj{select}};
		s/\bsrc\b/tra/i foreach @{$obj{where}};		
	}
	if ($params{cmd} =~ /all/) {
		unless (grep { /changedate/ } $params{tableRef}->fieldList) {	
			# if changedate does not exist, use date instead
			s/changedate/date/ and s/(.+)\sOR\s\1/$1/ foreach @{$obj{where}};			 
		}
		unless (grep { /date/ } $params{tableRef}->fieldList) { 
			# if date does not exist, use :txt instead : so that it can be binded even if we don't really use it
			s/date\s*(<>)?=/:txt =/ foreach @{$obj{where}};			 
		}		
	}
	$obj{select}[1] = '1 as score' if $params{cmd} =~ /all|version/; # not used
	return bless \%obj, $class;
}

sub _addMetaFields {
	my ($obj, $fields, $idField) = @_;
	if (grep { /((src|tra)_lang)|mem_name/i } @$fields) { 
		$obj->{from}[0] .= ' mem0'; s/(^|[^\.\:])(mem_(id|code|path))/mem0.$2/g foreach @{$obj->{where}}; 
		my $metaInfoView = config->{'meta-info'}{view} || 'public.meta_info'; $metaInfoView = "public.$metaInfoView" unless $metaInfoView =~ /\./;
		push (@{$obj->{from}}, "$metaInfoView meta"); push (@{$obj->{where}}, 'and' => "meta.$idField = mem0.$idField"); 
		push (@{$obj->{select}}, "meta.table_schema || '.' || meta.table_name as mem_name") if grep { /mem_name/i } @$fields;
		push (@{$obj->{select}}, "meta.$_") foreach grep { /_lang/i } @$fields;
		return 1;
	}
	return 0;
}

=head2 $newQuery = $query->addParam_displayScore ($spec)

Specify which algorithm to be used to calculate the score

=cut
sub addParam_displayScore {
	my ($self, $algo, $tags) = @_; $tags ||= $self->{tags};
	my %obj = %$self;
	$algo .= '(src,:txt)' if $algo =~ /\p{Letter}/ and $algo !~ /\(/;
	if ($algo =~ /tokenize/) {
		my $lang = undef; $lang = $obj{langSrc} || $obj{lang}; $lang = lc (substr($lang, 0, 2));		
		$algo =~ s/tokenize\s*\)/tokenize(SRC),tokenize(:txt))/;
		$algo =~ s/tokenize\s*\(([^,]+?)\)/tokenize($1,:srcStemmer,true)/g;
	}	
	$algo =~ s/untag\s*\)/untag(SRC),untag(:txt))/;
	$algo =~ s/untag\s*,/untag(SRC),/; $algo =~ s/untag\s*,/untag(:txt),/;
	$algo =~ s/untag\s*\(([^,]+?)\)/regexp_replace($1,E'$tags'::text, ''::text,'g')/g; # definition
	$algo =~ s!\((.+)\)!my $s = $1; my $t = $s; $t =~ s/SRC/:txt/; "($s,$t)"!e if ($algo =~ /SRC/ and $algo !~ /:txt/);	
	if ($obj{cmd} =~ /Tra/) { $algo =~ s/\bSRC\b/TRA/i; $algo =~ s/\bsrcStemmer\b/traStemmer/i; } 
	$obj{displayScore} = $algo;
	if (ref($obj{select})) { $obj{select} = [ map { my $copy = $_; $copy =~ s/^(.+) as score$/$obj{displayScore} as score/; $copy } @{$obj{select}} ]; } # 1-depth copy+replace
	if (ref($obj{with_sql})) { $obj{with_sql} = $obj{with_sql}->addParam_displayScore($obj{with_sql}, $algo); } # new object
	return bless \%obj, ref($self);
}

=head2 $newQuery = $query->addParam_filterContents ($spec)

Specify an algorithm to be applied to the contents (source or translation) before comparing to input in the where clause

=cut
sub addParam_filterContents {
	my ($self, $algo, $tags) = @_; $tags ||= $self->{tags};
	my %obj = %$self; foreach my $where (@{$obj{where}}) {
		$where =~ s/\b(src|tra)\b/regexp_replace($1,E'$tags'::text, ''::text,'g')/gi if $algo =~ /untag/;
		$where =~ s/\b(src|tra)\b/tokenize($1,:${1}Stemmer,true)/gi if $algo =~ /tokenize/;
	}
	$obj{filterContents} = $algo;
	if (ref($obj{with_sql})) { $obj{with_sql} = $obj{with_sql}->addParam_filterContents($obj{with_sql}, $algo); } # new object	
	return bless \%obj, ref($self);
}

=head2 $newQuery = $query->addParam_filterInput ($spec)

Specify an algorithm to be applied to the input before comparing to contents (source or translation) in the where clause

=cut
sub addParam_filterInput {
	my ($self, $algo, $tags) = @_; $tags ||= $self->{tags};
	my %obj = %$self; foreach my $where (@{$obj{where}}) {
		$where =~ s/:txt/regexp_replace(:txt,E'$tags'::text, ''::text,'g')/gi if $algo =~ /untag/;
		$where =~ s/:txt/tokenize(:txt,:srcStemmer,true)/ if $algo =~ /tokenize/;
	}
	$obj{filterInput} = $algo;
	if (ref($obj{with_sql})) { $obj{with_sql} = $obj{with_sql}->addParam_filterInput($obj{with_sql}, $algo); } # new object	
	return bless \%obj, ref($self);
}

=head2 $newQuery = $query->addParam_lang ($value, 's' (source) | 't' (translation, target) ?)

Specify that the query uses this language as source or target. Glossary queries may use it to find correct index.

=cut
sub addParam_lang {
	my ($self, $lang, $config, $type) = @_;
	unless ($type) {
		return $self->addParam_lang($self, $lang, $config, 'Src')->addParam_lang ($self, $lang, $config, 'Tra');
	}
	
	my %obj = %$self; our %HLANGS;
	$obj{"lang_$type"} = $lang;
	if ($obj{bind}) { $obj{bind} = { %{$obj{bind}} }; } # clone to modify
	my $lexer = config->{dict}{search}{$lang};
	unless ($lexer) { $lexer = config->{dict}{search}{substr($lang,0,2)}; }
	$lexer ||= $HLANGS{$lang} || 'simple'; # Pre-built default snowball configurations	
	$obj{bind}{':' . $type . 'Lexer'} = $lexer;		
	my $stemmer = config->{dict}{stem}{$lang} || config->{dict}{stem}{substr($lang,0,2)};
	$stemmer .= "_$lang" if $stemmer eq 'ispell'; $stemmer =~ s/\-/_/;
	$lang = lc (substr($lang, 0, 2));
	$stemmer ||= $HLANGS{$lang} . '_stem' if $HLANGS{$lang}; $stemmer ||= 'simple';
	$obj{bind}{':' . $type . 'Stemmer'} = $stemmer;	
	if (($obj{tableRef}{lc($type . '_lang')} =~ /\#V/) and ref($obj{tableRef}{children})) {	# browse children to restrict for this language
		require Silvestris::Cyclotis::Database::Table unless Silvestris::Cyclotis::Database::Table->can('find');
		$obj{where} = [ @{$obj{where}} ]; # will be modified
		my %byId = (); foreach my $table (values (%Silvestris::Cyclotis::Database::Table::CACHE)) {
			my $id = $table->{mem_id}; $id = $id->[0] if ref($id);
			$byId{$id} = $table;
		}
		my @keys = split(/,/, $obj{bind}{':childListTxt'}); $keys[0] =~ s/\{//; $keys[-1] =~ s/\}//;
		@keys = grep { $byId{$_}->{lc($type . '_lang')} =~ /\#V/ or $byId{$_}->{lc($type . '_lang')} =~ /^$lang/ } @keys;
		$obj{bind}{':childListTxt'} = sprintf('{%s}', join(',',@keys));		
	} elsif (($obj{tableRef}{lc($type . 'Lang')} =~ /\#V/) and ($obj{tableRef}{mem_path} =~ m!/!)) {	# lang restriction is in the mem path
		$obj{where} = [ @{$obj{where}} ]; # will be modified
		push (@{$obj{where}}, 'and', "mem_path ~ '\\($obj{lang_src}.*,'") if $type =~ /src/;
		push (@{$obj{where}}, 'and', "mem_path ~ ',$obj{lang_tra}.*\\)'") if $type =~ /tra/;
	}
	return bless \%obj, ref($self);
}


=head2 $newQuery = $query->uniq (@fields)

Filter returned lines using a list of fields: if two lines have the same values for given fields, only the last one (according date) is returned

Here you can give an array, an array ref or a specification as in C<Silvestris::Format::line>

=cut
sub addParam_uniq {
	my $self = shift; my %obj = %$self;
	my @uniqFields = @_;
	@uniqFields = @{$_[0]} if ref($_[0]);
	@uniqFields = split(/,/, $_[0]) if @uniqFields == 1 and $_[0] =~ /,/;

	my @fields = keys (%{$self->{tableRef}->{fields}}); 
	if (grep { /date/ } @fields) { # almost one date, we try to return most recent line
		$obj{with_name} = 'tmp'; $obj{with_sql} = new Silvestris::Cyclotis::Database::Query::Search(%$self);
		$obj{with_sql}{select} = [];
		foreach my $field (@uniqFields) { push(@{$obj{with_sql}{select}}, $self->{tableRef}->null_or_default($field) . " as $field"); }	
		if (grep { /changedate/ } @fields) { 
			if ($self->{tableRef}->{changedate} =~ /\?$/) { push(@{$obj{with_sql}{select}}, "max(coalesce(changedate,date)) as maxdate"); } 
			else { push(@{$obj{with_sql}{select}}, "max(changedate) as maxdate"); } # not nullable
		} else { 
			push(@{$obj{with_sql}{select}},  "max(date) as maxdate"); 
		}
		$obj{with_sql}{group_by} = \@uniqFields;
		# now with_sql is ready, we modify the other fields
		s/(\*|\w+,)/ori.$1/ foreach @{$obj{select}};
		foreach my $from (@{$obj{from}}) { $from .= ' ori' if ($from eq $self->{tableName}) or ($from eq $self->{std_parent}); }
		push (@{$obj{from}}, $obj{with_name});
		$obj{where} = [ map { ($self->{tableRef}->null_or_default("ori.$_") . " = tmp.$_", 'and') } @uniqFields ];
		if (grep { /changedate/ } @fields) { push(@{$obj{where}}, "(ori.changedate = tmp.maxdate or ori.date = tmp.maxdate)"); } 
		else { push(@{$obj{where}}, "ori.date = tmp.maxdate"); }
	} else {	# we use max(...) for each column, including textual ones
		$obj{select} = [ "$obj{displayScore} as score" ]; foreach my $field (@fields) {
			if (grep { $_ eq $field } @uniqFields) { push(@{$obj{select}}, $field); }
			else { push(@{$obj{select}}, "MAX($field)"); }
		}
		$obj{group_by} = \@uniqFields;
	}
	$obj{uniq} = \@uniqFields; return bless \%obj, ref($self);
}

=head2 $newQuery = $query->addParam_version ($date)

Returns all segments valid at the given date. The table must be versionned.

=cut
sub addParam_version {
	my ($self, $date) = @_;
	my %obj = %$self; our %CMD;
	my $verCmd = $CMD{version}; $verCmd =~ s/:txt/:version/g; 
	$obj{where} = [ $verCmd, 'and', @{$obj{where}} ]; # new array
	if ($obj{bind}) { $obj{bind} = { %{$obj{bind}} }; } # clone to modify	
	$obj{bind}{':version'} = $date;
	return bless \%obj, ref($self);
}

=head2 $newQuery = $query->addParam_sort (@criteria)

Sort by some columns. 

Here you can give an array, an array ref or columns separated by ','

For each of them, you must choose between ascending or descending order by sign + or -

=cut
sub addParam_sort {
	my $self = shift; 
	my @sortFields = @_;
	@sortFields = @{$_[0]} if ref($_[0]);
	@sortFields = split(/,/, $_[0]) if @sortFields == 1 and $_[0] =~ /,/;
	@sortFields = grep {	# do not sort on constant field, such as constant score
		my $fieldName = $1 if /(\w+)/;
		! ( grep { /\b\d+(\.\d+)?\s+as\s+\b$fieldName\b/ } @{$self->{select}} )
	} @sortFields; return $self unless @sortFields;
	my %obj = %$self; $obj{order_by} = [ map {
		substr($_,1) . (/^\+/ ?  ' ASC' : ' DESC')
	} @sortFields ];
	return bless \%obj, ref($self);
}

=head2 $newQuery = $query->addParam_page ($size, $id)

Returns page number 'id' (starting from 1) of size 'size'. 'id' is optional.
Please note that because the API is stateless, we do not guarantee that next call to the same query with next page will give the associated results.

=cut
sub addParam_page {
	my ($self, $size, $id) = @_;
	my %obj = %$self;
	$id ||= 1; my $offset = ($id - 1) * $size; 
	if ($obj{bind}) { $obj{bind} = { %{$obj{bind}} }; } # clone to modify	
	$obj{bind}{':limit'} = $size; $obj{bind}{':offset'} = $offset if $id > 1;
 	return bless \%obj, ref($self);
}

=head2 $string = $query->toSQL

Return the SQL as a string. 

Note that if you use DBI interface with result of this, you have to bind variables yourself. On the contrary, method "execute" does it for you.

=cut
sub toSQL { 
	my $self = shift;
	my $sql = 'select ' . join(', ', @{$self->{select}}) 
		. ' from ' . join(', ', @{$self->{from}})
		. ' where ' . join(' ', @{$self->{where}});
	if ($self->{with_name} && $self->{with_sql}) {
		my $with_sql = $self->{with_sql}; $with_sql = $with_sql->toSQL if ref($with_sql);
		$sql = "with $self->{with_name} as ($with_sql)\n$sql";
	}
	if ($self->{sqlConfig}{explicit}) {
		if (grep { /date|time/ } @{$self->{sqlConfig}{explicit}}) {
			$sql =~ s/(date)\s*([<>=]+)\s*(:\w+)/$1 $2 to_timestamp\($3,'YYYY-MM-DD HH24:MI:SS'\)/gs;
		}
		if (grep { /ts[\-\_]?vector/ } @{$self->{sqlConfig}{explicit}}) {
			# also implies ts_query, because we must ensure both use the same lexer
			$sql =~ s/(\w+)\s*\@\@\s*(:\w+)/sprintf("to_tsvector(:%s,%s) \@\@ plainto_tsquery(:%s,%s)", $1 . 'Lexer', $1, $1 . 'Lexer', $2)/ges;
		}
		elsif (grep { /ts[\-\_]?query/ } @{$self->{sqlConfig}{explicit}}) {
			$sql =~ s/\@\@\s*(:\w+)/\@\@ plainto_tsquery($1)/gs;	# in this case, use default lexer
		}
	}
	# if the query contains a lexer or a stemmer, ensure it will contain something even if we did not call addParam_lang
	$self->{bind}{':' . $1 . 'Lexer'} ||= 'simple' while $sql =~ /:(src|tra)Lexer/g;
	$self->{bind}{':' . $1 . 'Stemmer'} ||= 'simple' while $sql =~ /:(src|tra)Stemmer/g;
	# Special case : XOR in WHERE query (does not exist in SQL, replace by a couple of WITH queries
	while ($sql =~ /\bXOR\b/) {
		my $left = $1 if $sql =~ m!<left>(.+)</left>\s+XOR!; my $right = $1 if $sql =~ m!XOR\s+<right>(.+)</right>!; 
		my $queryLeft = $sql; $queryLeft =~ s!XOR\s+<right>(.+)</right>!!; $queryLeft =~ s!</?left>!!g;
		my $queryRight = $sql; $queryRight =~ s!<left>(.+)</left>\s+XOR!!; $queryRight =~ s!</?right>!!g;
		$sql = "with queryLeft as ($queryLeft), queryRight as ($queryRight)
					SELECT * FROM queryLeft
				UNION ALL
					SELECT * FROM queryRight
					 WHERE NOT EXISTS ( SELECT NULL FROM queryLeft )";         
	}
	# xOR between tables 
	if ($self->{tableName} =~ m!<!) {
		my ($tableParent, $tableChild) = split(m!<!, $self->{tableName});
		my $sqlChild = $sql; $sqlChild =~ s!$self->{tableName}!$tableChild!gs;
		my $sqlParent = $sql; unless ($sqlParent =~ s!$self->{tableName}!$tableParent!gs) {
			if ($sqlParent =~ s!mem_code\s*>?=\s*:mem_code(\s*and\s*mem_code\s*<=?\s*:max_code)?!mem_code\s*>=\s*:par_mem_code and mem_code <= :par_max_code!gs) {
				$self->{bind}{':par_mem_code'} = $self->{fallbackParentRef}{mem_code};
				$self->{bind}{':par_max_code'} = $self->{fallbackParentRef}{max_code};
			}
			if ($sqlParent =~ s!mem_path\s*>?=\s*:mem_path1?(\s*and\s*mem_path\s*<=?\s*:mem_path2)?!mem_path\s*>=\s*:par_mem_path1 and mem_path <= :par_mem_path2!gs) {
				$self->{bind}{':par_mem_path1'} = $self->{bind}{':par_mem_path2'} = $self->{fallbackParentRef}{mem_path};
				$self->{bind}{':par_mem_path2'} =~ s/(.)$/chr(ord($1) + 1)/e;
			}
		}
		$sql = "WITH queryParent AS ($sqlParent), queryChild AS ($sqlChild)
			(SELECT * FROM queryChild) UNION ALL (SELECT * FROM queryParent WHERE NOT EXISTS (SELECT null FROM queryChild))";
	}
	# Add sort and limit at last moment, because we sort the full result, not each subquery individually
	if ($self->{group_by} or $self->{order_by} or $self->{bind}{':limit'} or $self->{bind}{':offset'}) {
		$sql = "select * from ($sql) req " if $sql =~ /UNION/;
		$sql .= ' group by ' . join(', ', @{$self->{group_by}}) if $self->{group_by};
		$sql .= ' order by ' . join(', ', @{$self->{order_by}}) if $self->{order_by};
		$sql .= " LIMIT :limit" if $self->{bind}{':limit'}; $sql .= " OFFSET :offset" if $self->{bind}{':offset'};
	}
	# ok, now we return the result
	return $sql;
}

=head2 $dbi_statement = $self->execute($dbi_connection, $text)

Prepare the DBI statement, execute with $text and previously binded values.

=cut
sub execute {  
	my ($self, $dbh, $text) = @_; 
	my $sql = $self->toSQL(); my $st = $dbh->prepare_cached ($sql) or die $DBI::errstr;
	while (my ($k, $v) = each (%{$self->{bind}})) {
		if ($sql =~ /$k/) {
			$st->bind_param ($k => $v) or die $DBI::errstr;
		}
	}
    if ($self->{cmd} =~ /^glos/) { # Glossary search : use operator OR instead of AND
       my @TAB = ($text =~ /\p{Letter}+/g); 
       my %TAB = (); $TAB{lc($_)}++ foreach (@TAB); @TAB = keys(%TAB);     # Removes duplicates
       $text = join(' | ', @TAB);
    }	
	$st->bind_param(':txt', $text);
	$st->execute() or die $DBI::errstr;
	return $st;
}

=head2 $sql = $self->sqlExpressionForCommand ($cmd, $tableRef)

Returns the expression for the given search mode. Given parameter must be one of the following values.

Table reference is only used to check if note field exists.

=cut
sub sqlExpressionForCommand {
	my $self = shift; my $cmd = shift;
	if ($cmd =~ /exactOr(\w+)/) {
		my $sqlOther = $self->sqlExpressionForCommand(lc($1)); my $sqlExact = $self->sqlExpressionForCommand('exact'); 
		return "<left>$sqlExact</left> XOR <right>$sqlOther</right>"  # will be rewritten in toSQL
			if defined $sqlOther;
	}
	return $CMD{$cmd} if $CMD{$cmd} and $cmd !~ /Field/;
	if ($cmd =~ /^([a-z]+)(Src|Tra|Note|All|Any|SegAll|SegAny)$/) {
		my $mode = $1; my $fields = $2;
		my $pattern = $CMD{$mode . 'Field'} or return undef;
		if ($fields =~ /^(Src|Tra|Note)$/) { my $name = $1; $pattern =~ s/field/$name/g; return $pattern; }
		my @list = ('src','tra'); unless ($fields =~ /^Seg/) { my $tableRef = shift; push (@list, 'note') if ref($tableRef) and defined $tableRef->{fields}{note}; }
		@list = map { my $copy = $pattern; my $item = $_; $copy =~ s/field/$item/g; "($copy)" } @list;
		if ($fields =~ /Any/) { return join(' or ',@list); } else { return join(' and ', @list); }
	}
	return undef;
}

=head1 SEARCH MODES (i.e. possible values for C<:cmd> parameter)

=over 1

=line-item * Text modes: in these modes, query is a string.

=over 2

=line-item exact:		Searches for the exact given string in the source (i.e. the source must be I<exactly> identical to the given text)
=line-item like:		Search similar to SQL command C<like> in the source
=line-item fuzzy:		Linguistic fuzzy search in the source: the source must match the given text according to levenshtein distance
=line-item exactOrFuzzy:		Does exact search, then if no result found, does fuzzy search


=line-item contains(Src,Tra,Note,All,Any,SegAll,SegAny):	Returns all segments whose corresponding field contains the given text, as string without any analysis.

C<All> means searching in all three fields (or two, if note does not exist), with AND operator. C<Any> means searching in all three fields, with OR operator.  
C<SegAll> and C<SegAny> have same meaning but with only source and target, without note.

=line-item icontains(Src,Tra,Note,All,Any,SegAll,SegAny):	Identical to previous command but does case-insensitive search. Fields have same meanings than in previous command.

=line-item regex(Src,Tra,Note,All,Any,SegAll,SegAny):		Searches for the regular expression in the corresponding fields. Fields have same meanings than in previous command.

=line-item iregex(Src,Tra,Note,All,Any,SegAll,SegAny):		Searches for the regular expression, case-insensitive, in the corresponding fields. Fields have same meanings than in previous command.

=line-item concSrc,concTra:		Concordance (search for a sub-phrase) in the source (concSrc) or translation (concTra)

First, the terms are reduced as stems (roots) using the correct stemmer for the source language. 
Then the string must contain all the stems (except stopwords). Order is not checked.

=line-item glosSrc:		Terminology search (search for I<any> similar stem) in the source

First, the terms are reduced as stems (roots) using the correct stemmer for the source language. Then the string must contain almost one of the given terms.
The query is a phrase, but the database should contain only terms, i.e. small expressions without punctuation.

=back

=line-item * Date modes: in these modes, query is a date.

=over 2

=line-item all:			Search for all segments registered or updated after a given date.

Note: if the table is versionned, this returns all events after the given date, including deletions,
so if changedate is null it means that the segment is still valid, if not it is deleted.

=line-item version:		Returns all segments valid at the given date.

The table must be versionned (i.e. implement the 'version rules', look at dbcreator.pl for details)

=back

=back

=cut
our %CMD = (
    # Text search modes 
    fuzzy => 'src % :txt', exact => 'src = :txt', like => 'src like :txt', 
    # Partial text search modes
    containsField => 'position(:txt in field) > 0', icontainsField => 'position(lower(:txt) in lower(field)) > 0',
    regexField => 'field ~ :txt', iregexField => 'field ~* :txt',
    # Date search modes
    all => 'date >= :txt OR changedate >= :txt',
    version => 'date <= :txt and ((changedate is null) or (changedate >= :txt))', 
    # Stemmed string searches (concordance)
	concSrc => 'src @@ :txt', concTra => 'tra @@ :txt',
	glosSrc => "src @@ to_tsquery(:srcLexer,:txt)"
);

# ISO-639-1 language codes to Postgresql's tsvector english names
our %HLANGS = ( en => 'english', es => 'spanish', fr => 'french', de => 'german', da => 'danish', nl => 'dutch', fi => 'finnish',
                hu => 'hungarian', it => 'italian', nb => 'norwegian', nn => 'norwegian', no => 'norwegian', pt => 'portugese', 
                ro => 'romanian', ru => 'russian', sv => 'swedish', tr => 'turkish' );

1;

=head1 LICENSE

Copyright 2014-2018 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or  as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
