=head2 Serialization formats
 
These formats are standard structures to display data. However the list of available fields is not a standard and there is no way to describe it.

=head3 Available parameters

=over 1

=item envelope

Boolean, defaults to 1 (true). If set to 0/false, only the results array will be returned, without any meta-information such as query, tool version, and so on.

=item fields

Specify which fields (columns) of each line should be in the result. Default is to return fields from the table itself, but not meta-fields (table name and languages)

=item meta

Indicates that you want some meta-fields. This is a shortcut for parameter fields, where you only have to specify meta_name, src_lang and/or tra_lang
(while parameters fields requires to specify everything), or you can set meta=all to add all the meta-fields.

=back

=head3 Available serialization formats:

=cut
package Silvestris::Cyclotis::Format::nodep::serial;
use parent 'Silvestris::Cyclotis::Format';

sub new {
	my ($class, %params) = @_;
	if (my @tab = $class->buildFieldsList($params{lc(substr($class,rindex('::',$class) + 2)) . '.fields'} || $params{fields}, 0)) { $params{fields} = \@tab; } else { undef $params{fields}; }
	return bless \%params, $class;
}

sub dump_struct {
	my ($self, $struct, $indent, $fields) = @_;
	if ((ref($struct) =~ /HASH/) or ("$struct" =~ /HASH/)) {
	   my @fields = keys (%$struct); @fields = @$fields if $fields;
       if ($self->{indent} > 1) {	# 2 or 3
          my $prefix = "   " x ($indent + 2); my $res = "{\n$prefix   ";
		  if ($self->{indent} == 2) {
			$self->{indent} = 0;
			$res .= join(",\n$prefix   ", map { $self->dump_key_val ($_ => $$struct{$_}) }  @fields); 
			$self->{indent} = 2;
		  } else {	# 3 or more
			$res .= join(",\n$prefix   ", map { $self->dump_key_val ($_ => $$struct{$_}, $indent + 1) }  @fields); 
		  }
		  $res .= "\n$prefix\}"; return $res;
        } else {
          return '{' . join(',', map { $self->dump_key_val ($_ => $$struct{$_}) }  @fields) . '}';
        }
	} elsif ((ref($struct) =~ /ARRAY/) or ("$struct" =~ /ARRAY/)) {
        if ($self->{indent} > 1) {
          my $prefix = "   " x ($indent + 2); my $res = "[\n$prefix   ";
		  if ($self->{indent} == 2) {	# 2 or 3
			$self->{indent} = 0;
			$res .= join(",\n$prefix   ", map { $self->dump_struct($_) }  @$struct); 
			$self->{indent} = 2;
		  } else { # 3 or more
			$res .= join(",\n$prefix   ", map { $self->dump_struct($_, $indent + 1) } @$struct);
		  }
		  $res .= "\n$prefix\]"; return $res;
        } else {
          return '[' . join(',', map { $self->dump_struct($_) } @$struct) . ']';
        }
	} elsif ($struct =~ /^[\+\-]?(\d+)(\.\d+)?(E[\+\-]?\d+)?$/) {
		return $struct;	# is a number
	} else {
		$struct =~ s/\"/\\\"/gs;
		return qq("$struct");	# is a string
	}
}

# json result is an array
sub header { 
	my $self = shift; my %params = @_;
	unless (defined ($self->{envelope}) && ($self->{envelope} == 0)) {
		my %struct = (status => 'OK', info => { query => $params{query} }, result => []);
		if ($self->{'display-date'}) { my @t = localtime; $t[5] += 1900; $t[4]++; $struct{info}{date} = "$t[5]-$t[4]-$t[3] $t[2]:$t[1]:$t[0]"; }
		my $ref = ref($self) || $self; $ref = substr($ref,rindex($ref,'::'));
		if ($self->{verinfo}) { my %v = $self->VERSION_INFO(); $struct{info}{cyclotis} = { version => $v{main}, "format-$ref"  => $v{fmt} }; }
		my $str = $self->dump_struct(\%struct);
	} else {
		if ($self->{comments} == 0) { "[\n" } else { "[\n   " . $self->comment($_[0], 1) . "   " }
	}
}
sub footer { 
	my $self = shift; my $count = shift;
	unless (defined ($self->{envelope}) && ($self->{envelope} == 0)) {
		"   ],\n" . $self->dump_key_val(result_count => $count) . "\n}"
	} else {
		if ($self->{comments} == 0) { "]" } else { $self->comment("$count results", 0) . " ]" }
	}
}

sub produce_info { 
	my $self = shift; 
	unless (defined ($self->{envelope}) and $self->{envelope} == 0) {
		my $res = $self->header(query => 'info') . $self->footer(); 
		my @tables; @tables = split(/,/, $_[-1]) if $_[-1] =~ /,/;
		$res =~ s!\},!$& . $self->dump_key_val(config => { formats => shift, url => shift, allow => shift, timestamp => shift, tables => { read => $tables[0], write => $tables[1] } }) . ','!e;
		return $res;
	} else {
		return $self->dump_key_val(config => { formats => shift, url => shift, allow => shift });
	}
}

sub struct_info {
	my ($self, %tables) = @_;
	unless (defined ($self->{envelope}) and $self->{envelope} == 0) {
		my $res = $self->header('info') . $self->footer(); 
		$res =~ s!\},!$& . $self->dump_key_val(tables => \%tables) . ','!e;
		return $res;
	} else {
		return $self->dump_struct (\%tables);
	}
}

sub produce_line { 
	my ($self, $line) = @_; my $dump = $self->dump_struct ($line, undef, $self->{fields});
    if ($self->{indent}) { return "      $dump,\n"; } else { return "$dump,"; }
}

sub produce_error { 
	my $err = $_[1]; 
	if (ref($err) and $err->isa('Dancer::Error')) { 
		return $_[0]->dump_struct ({ status => "error", code => $err->code, message => $err->message });
	} else {
		return $_[0]->dump_struct ({ status => "error", message => $err });
	}
}

sub status_line { 
	my ($self,$line) = @_; my $res = { %$line }; delete $res->{dbi}; $res->{insertions} = $line->{dbi};
	return $self->dump_struct($res) . ',';
}

sub show_status {
	my ($self, $level, $types, $Queries, $Lib) = @_; my %res;
	unless ($types !~ /cache/) {
		$res{cache}{tables} = scalar keys (%Silvestris::Cyclotis::Database::Table::CACHE); 
		$res{cache}{queries} = scalar @$Queries; 
		if ($level > 1) { 
			$res{cache}{tables} = { count => $res{cache}{tables}, items => [keys (%Silvestris::Cyclotis::Database::Table::CACHE)] }; 
			$res{cache}{queries} = { count => scalar (@$Queries), items => $Queries }; 
		}
	}
	unless ($types !~ /libs/) {
		$res{libs} = $Lib;
	}
	return $self->dump_struct(\%res);
}

sub show_log {
	my $self = shift; my $log = shift; return $self->dump_struct($log);  
}

sub needs_field { 
	if (defined $_[0]->{fields}) { 
		if (ref ($_[0]->{fields})) { return grep { /$_[1]/ } @{$_[0]->{fields}}; } # as an array
		else { return ($_[0]->{fields} =~ /$_[1]/); }	# comma-separated string
	}
	if (defined $_[0]->{meta}) {  
		if (ref ($_[0]->{meta})) { return grep { /$_[1]/ } @{$_[0]->{meta}}; } # as an array
		else { return ($_[0]->{meta} =~ /$_[1]|all/); }	# comma-separated string
	}
	return $_[0]->SUPER::needs_field($_[1]);
}

sub openapi {
  my ($self, $indent, $config) = @_;
  require Silvestris::Dancer::MyRoutes unless Silvestris::Dancer::MyRoutes->can('openapi');
  my %struct = Silvestris::Dancer::MyRoutes->openapi;
  return $self->dump_struct(\%struct, $indent);
}

1;

=head1 LICENSE

Copyright 2013-2017 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or  as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
