=head1 DESCRIPTION

Adds Dancer commands specific to Silvestris tools

Even if it is written as a Dancer plugin, it contains specific methods which should not be used elsewhere.

=cut
package Silvestris::Dancer::MyRoutes;

use Dancer ':syntax';
use Dancer::Plugin;

require Silvestris::Cyclotis::Format if config->{'auth-db'}{'permissions'};

# our @ROUTES = (); # @Silvestris::Dancer::MyRoutes::ROUTES

sub createRecursiveRoutes {
	my ($methods, $conf, $destSub) = @_;
	if ($conf =~ m!\((.+)\)\?!) {
		# Routes for get. Longest version must come first!!!
		my $conf1 = $conf; $conf1 =~ s!\((.+?)\)\?!$1!;	# no 'g', only first instance
		my $conf2 = $conf; $conf2 =~ s!\((.+?)\)\?!!;	# no 'g', only first instance
		$conf1 =~ s/^([^\(]+)\)\??/$1/; $conf2 =~ s/^([^\(]+)\)\??/$1/; # supports ((something)?other)? : in case of ambiguity only "other" will be filled
		return (
			createRecursiveRoutes ( $methods, $conf1, $destSub),
			createRecursiveRoutes ( $methods, $conf2, $destSub)
		);
	} else {
		# Route is complete. create it
		any $methods => $conf => $destSub;
    foreach my $m (@$methods) { push(@Silvestris::Dancer::MyRoutes::ROUTES, "$m $conf"); } 
		return ($conf); # only one result
	}
}

# Recursive method to check if the user is allowed to do this action.
# Returns : undef if success, error message else.
sub forbid {
	my $require = shift;
	if (ref($require) =~ /ARRAY/) {		# ARRAY means "OR"
		my $err = undef;
		foreach (@$require) {
			$err = forbid($_);
			unless ($err) { return undef; }	# we found one condition which succeeded
		}
		return $err; # the last one
	} elsif (ref($require) =~ /HASH/) {		# HASH means "AND"			
		if (my $requiredGroup = $require->{group}) { my $res = forbid_user($requiredGroup, var 'auth_group'); return $res if $res; }	# list of accepted groups
		if (my $requireUser = $require->{user}) { my $res = forbid_user($requireUser, var 'auth_user'); return $res if $res; }	# list of accepted users
		if (my $requireTable = $require->{table}) {		# list of tables, and/or can use * as a joker
			my $table = param('table');
			if (param('schema')) { $table = join('.', param('schema'), $table); }
			else { $table = "public.$table" unless $table =~ /\./; }
			$requireTable = [split(/,/,$requireTable)] unless ref $requireTable;
			my $user = var 'auth_user'; my $group = var 'auth_group';
			foreach (@$requireTable) { 
				s/\*/\\w\+/; s/\?/\\w/; # convert to regular expression
				s/:user/$user/; s/:group/$group/; 	# replace variables
			}
			return sprintf ('User %s cannot do :name on %s', (var 'auth_user'), $table) if forbid_list($requireTable, $table); 
		}
		if (my $requireCmd = $require->{cmd}) {		# list of commands. Can use ! for blacklist, but no jokers.
			$requireCmd = [$requireCmd] unless ref $requireCmd;	
			return sprintf ('Command %s is not permitted', param('cmd')) if forbid_list($requireCmd, param('cmd'));
		}
		# Rules which depend on meta_view's fields
		foreach my $meta (grep { /^meta\./ } keys (%$require)) {
			use Silvestris::Cyclotis::Database::Table;
			my $table = find Silvestris::Cyclotis::Database::Table (param('db'), param('table'));
			my $val = $table->{substr($meta,5)} or return undef; my @entries = (ref ($require->{$meta}) =~ /ARRAY/) ? @{$require->{$meta}} : (split (/,/, $require->{$meta}));
			s/\$([\w\-]+)/param($1) || ''/e foreach @entries; 	# replace variables with parameters
			s/\%([\w\-]+)/request->header($1) || ''/e foreach @entries; 	# replace variables with request headers	
			s/crypt\((.*?)\)/&$Silvestris::User::CryptAlgo($1)/e foreach @entries; 	# entries can be encrypted
			@entries = map { "^$_\$" } @entries;
			return "$meta for table $table->{table_name} does not correspond to given parameters" if forbid_list(\@entries, $val);
		}
	}
}

sub forbid_user {
	my ($requireUser, $currentUser) = @_; $requireUser = [$requireUser] unless ref $requireUser;
	unless (grep { $currentUser eq $_ } @$requireUser) {
		return sprintf('User %s is not in the white list for :name', (var 'auth_user'));			
	} else {
		return undef; # success
	}
}

sub forbid_list {
	my ($list, $val) = @_;
	if ($list->[0] =~ /^\!/) { 
		foreach my $t (@$list) { my $t1 = $t; $t1 =~ s/^\!//; return undef unless $t1 and $val =~ $t1; }	# black list 
	} else {
		foreach my $t (@$list) { return undef if $val =~ $t; }	# white list
	}
	return 1; # if none of foreach succeeded
}

=head1 Added perl-Dancer words

=head2 sv_route 'name' => \@methods => $defaultRoute => sub { ... };

Adds the routes to Dancer application. The differences with original methods:

=over 1

=item *

Routes can contain optional parts. For example /route(/:param1)? is equivalent to [ "/route", "/route/:param1" ]

=item *

If config->{url}{schemas}{name} is defined, it replaces $defaultRoute

=item *

In the default route (not in config), parameter :db is removed if multiple connections have not been configured.

=item *

If default route (before configuration) is empty (equal to "", not to "/") it means that by default this route is not available at all.
This is useful for deletion routes, which we may want to forbid by default.

=item *

If authorization is defined, the route will first check if the operation is permitted or not

=back

=cut
register 'sv_route' => sub {
	my ($name, $methods, $defaultRoute, $callback) = @_;
	unless (config->{plugins}{Database}{connections}) { $defaultRoute =~ s!/?:db/?!!; $defaultRoute =~ s!\(\)\?!!; }
	my ($cat, $op) = split(/\./, $name);
	if ($cat and $op) { if (config->{url}{schemas}{$cat}{$op}) { $defaultRoute = config->{url}{schemas}{$cat}{$op}; } else { config->{url}{schemas}{$cat}{$op} = $defaultRoute; } }
	else { if (config->{url}{schemas}{$name}) { $defaultRoute = config->{url}{schemas}{$name}; } else { config->{url}{schemas}{$name} = $defaultRoute; } }
	return unless $defaultRoute; # Route whose default is "" means that by default we do not want to create it
	$defaultRoute = [$defaultRoute] unless ref $defaultRoute; $methods = [$methods] unless ref $methods;
	if (my $perm = config->{'auth-db'}{'permissions'}) {
     # first check for command-specific, then for global: here the array means AND, we do not want to repeat global restrictions
		$callback = restrict ($name, $callback, ($perm->{$name} || $perm->{$cat}{$op}, $perm->{$cat}{any} || $perm->{"$cat.any"}, $perm->{any}));
	}
	if (config->{'usage-log'}) { my $oriCallback = $callback; $callback = sub { var 'command' => $name; return &$oriCallback(); }; }
	foreach my $route0 (@$defaultRoute) {
		my @adds = createRecursiveRoutes ($methods, $route0, $callback);
		&{SV_debug ("routes")} (sprintf("Route for %s: %s %s", $name, join(',',@$methods), $_)) foreach @adds;
	}
};

=head2 $callback = restrict($name, $callback, @conditions)

If conditions are defined, encapsulate callback in a sub which will check them. Else, return callback itself

=cut
sub restrict {
  my ($name, $callback, @conditions) = @_; return $callback unless @conditions;
  my $condition0 = shift(@conditions); return restrict($name, $callback, @conditions) unless ref $condition0;
  # now we must do encapsulation
  return sub {
    if (my $forbid = forbid($condition0)) {
      status 401;
      $forbid =~ s!:name!$name!;
      my $fmt = Silvestris::Cyclotis::Format->for_dancer_params();
      halt $fmt->reencode (1, $fmt->produce_error($forbid));
    }
		return &$callback();
  };
}

=head2 my $function = sv_debug("category"); &$function("message1", "message2", ...);

=head2 my $function = sv_debug "category" => sub { some_filters(@_) }; &$function("message1", "message2", ...);

Returns a function to indicate where to put debug messages for the given category.

Once created, unless we use filters, the function is kept in cache for later calls.

=cut
our %SV_DEBUG;
sub SV_debug {
	my $spec = shift; $spec = config->{debug}{$spec} if config->{debug}{$spec}; 
	my $fun = shift; unless ($fun) { 	# can use cache
		return $SV_DEBUG{$spec} ||= SV_debug($spec, sub {@_}); # call with identity as a filter
	}
	if (ref($spec) =~ /ARRAY/) { my @procs = map { SV_debug($_, $fun) } @$spec; return sub { foreach my $p (@procs) { &{$p}(&$fun(@_)); } }; }
	if (($spec eq '*STDERR') or ($spec eq '2')) { return sub { print main::STDERR &$fun(@_), "\n"; } }
	if (($spec eq '*STDOUT') or ($spec eq '1')) { return sub { print main::STDOUT &$fun(@_), "\n"; } }
	if ($spec eq '&info') { return sub { info $_ foreach &$fun(@_); } }
	if ($spec eq '&debug') { return sub { debug $_ foreach &$fun(@_); } }
	return sub {};
}
register 'sv_debug' => \&SV_debug;



register_plugin for_versions => [ 1, 2 ];

=head2 my $ref = openapi ()

Returns a struct which can be converted to json or yaml for openapi

=cut
sub openapi {
  my %struct = (swagger => "2.0", info => { version => "2017.0", title => "Cyclotis Translation Memory contents API"}, host => `hostname` . ":5000", schemes => ["http"], paths => {});
  $struct{host} =~ s/\n//;
  foreach my $route (@Silvestris::Dancer::MyRoutes::ROUTES) {
    my ($m, $p) = split(/ /, $route); 
    my @params; push(@params, { name => $1, in => 'path', required => 'true', type => 'string' }) while $p =~ s/:(\w+)/"{$1}"/e;
    $struct{paths}{$p}{lc($m)} = { parameters => \@params };
  }
  return %struct;
}

1;

=head1 LICENSE

Copyright 2013-2017 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or - as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
