=encoding utf-8
=head1 DESCRIPTION

This package contains the output formats supported by Cyclotis

=head1 Available formats:

=cut
package Silvestris::Cyclotis::Format;

our @FIELDS = ('src','tra','author','date','score','changer','changedate','context','note','props');
our %sFIELDS = (o => 'src', t => 'tra', a => 'author', d => 'date', s => 'score', c => 'changer', u => 'changedate', 
             i => 'context', n => 'note', p => 'props', 'r' => 'row_id',
             # Meta-fields : needs jointure
             O => 'src_lang', T => 'tra_lang', N => 'mem_name');

sub buildFieldsList {
	my ($self, $spec, $mandatory) = @_; $spec ||= '';
	if (ref($spec) =~ /ARRAY/) { return @$spec; }
	elsif ($spec =~ /,/) { return split(/,/, $spec); }
	elsif ($spec =~ /^\$/) { return map { $sFIELDS{$_} } ($spec =~ /([a-zA-Z])/g); }
	elsif ($mandatory) { return @FIELDS; }
	else { return (); }
}

sub VERSION_INFO {
	my ($mainFile) = grep { /rest\-api\.pl$/ } values (%main::INC);

	my @cyStat = stat($mainFile); my @fmtStat = stat($INC{'Silvestris/Cyclotis/Format.pm'});
	@cyStat = localtime($cyStat[9]); $cyStat[5] += 1900; $cyStat[4]++;
	@fmtStat = localtime($fmtStat[9]); $fmtStat[5] += 1900; $fmtStat[4]++;
	return (main => sprintf('%4i-%02i-%02i %02i:%02i:%02i', @cyStat[5,4,3,2,1,0]), fmt => sprintf('%4i-%02i-%02i %02i:%02i:%02i', @fmtStat[5,4,3,2,1,0]));
}

use Silvestris::Dancer::MyRoutes;
our %cache;

sub new {
	my ($class, %params) = @_;
	
	my $debug = Silvestris::Dancer::MyRoutes::SV_debug('formats'); 
	my $name = $params{name} || substr($class, rindex($class,'::') + 2);
	if ($cache{$name}) { # Already used
		$class = $cache{$name};
	} elsif ($name =~ /(nodep|cpan)/i) { # Administrator explicitly required nodep or cpan format
		$cache{$name} = $class = "Silvestris::Cyclotis::Format::\l$1::$name";
		&$debug("Use $class for $name, as declared in config");
		unless ($class->can('produce')) { eval "require $class" or die $@; }
	} elsif ($name =~ /^e\w+$/) { 	# may be an enveloppe format
		eval "require Silvestris::Cyclotis::Format::nodep::Envelope" unless Silvestris::Cyclotis::Format::nodep::Envelope->can('produce');
		my $subFormat = substr($name,1); my $subClass = undef;
		foreach my $s ('cpan','nodep') {
			if ("Silvestris::Cyclotis::Format::${s}::$subFormat"->can('produce')) {	# already loaded format
				$subClass = "Silvestris::Cyclotis::Format::${s}::$subFormat";
			} elsif (eval "require Silvestris::Cyclotis::Format::${s}::$subFormat") {	# we could load it
				$subClass = "Silvestris::Cyclotis::Format::${s}::$subFormat";			
			}
			if ($subClass) {
				$subFormat = $subClass->new(%params, name => $subFormat, embedded => 1);
				return new Silvestris::Cyclotis::Format::nodep::Envelope (%params, subFormat => $subFormat);
			}
		}
	} elsif ($name eq 'xml') {	# works only for struct, info
		$class = $cache{$name} = 'Silvestris::Cyclotis::Format::nodep::Envelope';
		eval "require Silvestris::Cyclotis::Format::nodep::Envelope" unless Silvestris::Cyclotis::Format::nodep::Envelope->can('produce');
	} else {
		if (eval "require Silvestris::Cyclotis::Format::cpan::$name") {
			# True if exists _AND_ we succeeded to load it, so the dependencies are present.
			$cache{$name} = $class = "Silvestris::Cyclotis::Format::cpan::$name";
			&$debug("Use $class for $name, as cpan module works");
		} else {
			$cache{$name} = $class = "Silvestris::Cyclotis::Format::nodep::$name";
			unless ($class->can('produce')) { eval "require $class" or die $@; }
			&$debug("Use $class for $name, as cpan module does not work ($@)");
		}	
	}
	
	return $class->new(%params);
}

use Dancer qw(param params config content_type request);
require POSIX if grep { config->{$_}{'date-format'} } keys %{config->{formats}};

sub for_dancer_params {
	shift if $_[0] eq 'Silvestris::Cyclotis::Format';	
	my $fmt = shift || param('fmt');
	unless ($fmt) {
		my $env = request->env;
		if (my $accept = $env->{'HTTP_ACCEPT'}) {
			my @accept = split(',', $accept);
			ALOOP: while (my $a = shift(@accept)) { last ALOOP if $fmt = Silvestris::Cyclotis::Format->for_mime($a); }
		} elsif ((request->user_agent || '') =~ /Mozilla/i) {
			$fmt = 'html';
		}
		$fmt ||= config->{formats}{default} || 'line';
	}
 	my %params = (); %params = %{config->{formats}{'all'}} if config->{formats}{'all'}; %params = (%params, %{config->{formats}{$fmt}}) if config->{formats}{$fmt};
	%params = (%params, %{config->{formats}{'Envelope'}}) if $fmt =~ /^e/ or $fmt =~ /Envelope/;
	%params = (%params, params('query')); delete @params{'cmd','query','db','table','schema'};
	$params{indent} = 3 if ((request->user_agent || '') =~ /Mozilla/i) and (not defined $params{indent});
	return new Silvestris::Cyclotis::Format(%params, name => $fmt);
}

sub reencode {
	my $self = shift; my $addHeader = shift; my @res; while (my $res = shift) {
		if (my $code = $self->{encoding}) { 
			content_type ($self->MIME() . "; charset=$code") if $addHeader;
			push(@res, Encode::encode($code, Encode::decode_utf8($res))); 
		} else { 
			content_type ($self->MIME() . "; charset=utf-8") if $addHeader;
			push(@res, Encode::encode_utf8($res));	# utf-8, but Plack wants bytes, not characters.
		}
		$addHeader = 0;
	}
	return join('',@res); # Dancer wants a scalar.
}

# Method which should be correct for most formats
sub reformat_dates {
    my ($self, $line) = @_;
    if ($self->{'date-format'}) {
		foreach (grep { /date/ } keys %$line) {
			my @t = split(/\D+/, $line->{$_}); $t[0] -= 1900; $t[1]--;	
			$line->{$_} = POSIX::strftime($self->{'date-format'}, @t[5,4,3,2,1,0]);
		}
	}
}

# Return the name to be passed to 'new' according to MIME
sub for_mime {
	my ($class, $name, %params) = @_;
	return 'line' if $name =~ /text/ and $name =~ /plain/;
	return 'json' if $name =~ /json/;
	return 'html' if $name =~ /html/;
	return 'etmx' if $name =~ /xml/;
	return 'tmx' if $name =~ /tmx/;
	return undef;
}

sub header { '' }
sub footer { '' }

# Optimisation : some formats may not use all fields, in which case it is better not to calculate them
sub needs_field { # default, to be overridden
	return $_[1] ne 'score' && $_[1] !~ /_/;		# by default, we exclude meta_fields and score
}


1;

=head1 LICENSE

Copyright 2013-2016 Silvestris Project (L<http://www.silvestris-lab.org/>)

Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
