=head2 Format 'yaml'

Returns a YAML document

=cut
package Silvestris::Cyclotis::Format::nodep::yaml;
use parent Silvestris::Cyclotis::Format::nodep::json;	# according to its sepecification, YAML is an extension of JSON

sub MIME { 'application/x-yaml' }

sub unquote { 
    my $txt = shift; $txt = shift if ref($txt) or $txt =~ /^Silvestris::/; 
    $txt =~ s/"(([^,\\]|(\\\"))+?)"/$1/gs; return $txt; 
}

sub footer { "...\n" }

sub header {
	my $self = shift; my %args = @_;
	return unquote($self->SUPER::header(@_)) if (defined $self->{indent}) and ($self->{indent} < 1);
	my $cy = ""; if ($self->{verinfo}) { my %v = $self->VERSION_INFO(); $cy = "cyclotis: {version: '$v{main}', format-yaml: $v{fmt}}"; }
	return << "EOF";
status: OK

info:
   query: $args{query}
   $cy

result:   
EOF
}

sub produce_info {
	my $self = shift; my $formats = shift; my $url = shift; my $timestamp = shift;
	my @tables = split(/,/, shift);
	my $res = $self->header(query => 'info'); $res .= "\n\n";
	$res .= << "EOF";
config:
   timestamp: $timestamp
   tables:
      read: $tables[0]
      write: $tables[1]
   formats:
      default: $formats->{default}
   url:
      struct: $url->{meta}{struct}
      find: $url->{contents}{find}
      save: $url->{contents}{save}
      delete: $url->{contents}{delete}

EOF
	return $res;
}

sub struct_info {
	my ($self, %tables) = @_;
	my $res = $self->header(); $res .= "\n\ntables:\n";
	while (my ($table, $spec) = each (%tables)) {
		$res .= "   $table:\n";
		$res .= "      cols:\n";
		while (my ($name, $type) = each(%{$spec->{fields}})) { $res .= "         $name: $type\n"; }
		$res .= "      options:\n";
		while (my ($name, $val) = each(%$spec)) { 
			next if $name =~ /fields/;
			if ($name =~ /props/) {
				$res .= "         $name:\n"; 
				while (my ($k, $v) = each (%$val)) { $res .= "            $k: $v\n"; }
			}
			$res .= "         $name: $val\n"; 
		}
		$res .= "\n";
	}
	return $res;
}

sub produce_line { 
    my ($self,$line) = @_; 
	if (defined $self->{indent}) {
		return unquote($self->SUPER::produce_line($line)) if (defined $self->{indent}) and ($self->{indent} < 1);
		if ($self->{indent} == 1) {
			return "   - " . unquote($self->SUPER::produce_line($line)) . "\n";			
		}
	}
	# indent = 2 or undef
    my $first = '-'; my $prefix = '   '; my $res = ''; my @fields = keys(%$line); if (my $fields = $self->{fields}) { @fields = @$fields; } 
    while (my $field0 = shift (@fields)) { 
		my $value = $line->{$field0} or next;
		$res .= $prefix . "$first $field0: "; $first = ' ';
		unless (ref ($$line{$field0})) { $res .= " $value\n"; }
		else {
			if ($self->{indent} == 3) {
				$res .= "\n";
				if (ref ($value) =~ /ARRAY/) {
					$res .= join("\n", map { "$prefix    - $_" } @$value);
					$res .= "\n";
				} elsif (ref ($$line{$field0}) =~ /HASH/) {
					$res .= join("\n", map { "$prefix    $_: $line->{$field0}{$_}" } keys(%$value));
					$res .= "\n";
				}
			} else {	# 2 is default
				$self->{indent} = 0;
				$res .= unquote($self->dump_struct ($value)) . "\n";
				$self->{indent} = 2;
			}
		}
	}
	return $res . "\n\n";
}

sub status_line { 
	my ($self,$line) = @_; my $res; 
	if ($line->{error}) { $res = "error: $line->{error}\n"; } else { $res = "insertions: $line->{dbi}"; }
	my $dump = $self->dump_struct($line); $dump =~ s/^(\s+)/$1$1/gm;
	return "   - $res\n    contents:\n$dump\n"; 
}

1;

=head1 LICENSE

Copyright 2013-2016 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or  as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
