#! /usr/bin/env plackup

=head1 DESCRIPTION

This file adds a hook to an existing Dancer application.
The hook is invoked before any query, it tries to authenticate based on a table in the database.
In case of failure an HTTP status 4xx/5xx and a message are generated, and the query is cancelled.
In case of success, it also sets variables auth_user and auth_group, so that the application can check for permissions.

=cut

use MIME::Base64;

use DBI;  
use Dancer::Plugin::Database;

use Silvestris::Cyclotis::Format;

my $authConf = config->{'auth-db'};

$Silvestris::User::CryptAlgo = sub { shift };	# by default, empty method
if ($authConf->{'password-format'} =~ /MD5/i) {
	require Digest::MD5; no strict; 
	$Silvestris::User::CryptAlgo = \&{"Digest::MD5::" . lc($authConf->{'password-format'})}	# may contain md5 or md5_hex
		or die "Password format $authConf->{'password-format'} does not exist";
} elsif ($authConf->{'password-format'} =~ /SHA/i) {
	require Digest::SHA; no strict; 
	$Silvestris::User::CryptAlgo = \&{"Digest::SHA::" . lc($authConf->{'password-format'})}
		or die "Password format $authConf->{'password-format'} does not exist";		
}
eval "require $1" if $authConf->{'column-transforms'}{password} =~ /^([\w\:]+)::(\w+?)$/;

use Dancer;

# ---- How to check if the user exists and password is correct: no default method because it may be delegated to proxy
# $checkMethod is a sub which answers (status > 400, error message) or (0, undef, user group)
my $checkMethod; my $checkParams = $authConf->{'check-method'};

if (! $checkParams->{type}) { #  no check. Useful for example if we are behind a proxy which already checked the password
	$checkMethod = sub { return (0, undef, undef); };
} elsif ($checkParams->{type} =~ /db/i) {	# authenticate using a database
	my $sql = "select * from $checkParams->{table} where $checkParams->{'column-names'}{user} = ?";
	if ($checkParams->{filters}) {
		foreach my $filter (@{$checkParams->{filters}}) {
			$sql .= " and $filter->{column} = ?";
		}
	}
	debug "Authentication SQL = $sql";
	my $dbh = $checkParams->{'dbi-url'} ? DBI->connect (@{$checkParams}{'dbi-url','dbi-user','dbi-pass'}) 	# database specific to authorization
		: database($checkParams->{'dbi-id'});	# try to use database plugin
	$sql = $dbh->prepare_cached ($sql);
	$checkMethod = sub {
		my ($user, $pass) = @_;
		$pass = &$Silvestris::User::CryptAlgo($pass);
		my @params = ($user);
		if ($checkParams->{filters}) {
			foreach my $filter (@{$checkParams->{filters}}) {
				my $value = $filter->{value};
				$value =~ s{:(\w+)}{
					my $name = $1;
					if (param($name)) { param($name) }
					elsif ($name eq 'schema') { param('table') =~ /^(\w+)\.(\w+)$/ ? $1 : 'public' }
					else { die "Could not find value for $name"; }
				}gex;
				push (@params, $value);
			}
		}
		unless ($sql->execute(@params)) {
			error "Could not execute authentication : $DBI::errstr";
			return (500, "Authentication not possible");
		} else {
			if (my $ref = $sql->fetchrow_hashref) {
				my $dbPass = $ref->{$checkParams->{'column-names'}{password}};
				if ($checkParams->{'column-transforms'}{password} eq 'uuid_to_hex') { $dbPass =~ s/\-//g; }
				elsif ($checkParams->{'column-transforms'}{password} =~ /^\s*sub/) { my $sub = eval $checkParams->{'column-transforms'}{password}; $dbPass = &$sub($dbPass); }
				elsif ($checkParams->{'column-transforms'}{password}) { $dbPass = &{$checkParams->{'column-transforms'}{password}} ($dbPass); }
				if ($dbPass ne $pass) {
					return (401, "Wrong  password for user '$user'");
				} else {
					# Authentication is OK. Now it is responsibility of the application to allow or not to do something
					my $group = $ref->{$checkParams->{'column-names'}{group}} if $checkParams->{'column-names'}{group};
					return (0, undef, $group); # Success
				}
			} else {
				return (401, "Unknown user : '$user'");
			}
		}
	};
} elsif ($checkParams->{type} =~ /url/) {		# call an authentication service which uses HTTP protocol
	require LWP::Simple;
	$checkMethod = sub {
		my ($user, $pass) = @_;
		$pass = &$Silvestris::User::CryptAlgo($pass) if $checkParams->{'crypt-pass'};
		my $url = $checkParams->{url}; $url =~ s/:user/$user/; $url =~ s/:pass/$pass/;
		my $res = LWP::Simple::get($url);
		if ($res =~ /$checkParams->{correct}/o) {
			my $group = $1; $group ||= $1 if $res =~ /$checkParams->{group}/o; return (0, undef, $group);
		} else {
			my $err = $1 if $res =~ /$checkParams->{error}/o; my $status = $2 || 401;
			return ($status, $err || "Wrong  password for user '$user'");
		}
	};
} elsif ($checkParams->{type} =~ /file/i) {		# authenticate using a text file
	open (F, $checkParams->{fileName}) or die "Cound not load authentication file: $!";
	my @LINES = <F>; close (F); @LINES = map { s/\r?\n//; [split($checkParams->{separator} || ':',$_) ] } @LINES;
	my %USERS = map { $_->[$checkParams->{columns}{user} || 0] => $_ } @LINES;
	$checkMethod = sub {
		my ($user, $pass) = @_;
		$pass = &$Silvestris::User::CryptAlgo($pass);
		if ($pass eq $USERS{$user}[$checkParams->{columns}{password} || 1]) {
			return (0, undef, $USERS{$user}[$checkParams->{columns}{group}]); # Success
		} elsif ($USERS{$user}) {
			return (401, "Wrong  password for user '$user'");
		} else {
			return (401, "Unknown user : '$user'");
		}
	};
} elsif ($checkParams->{type} =~ /config/i) {		# authenticate using parameters in the config file itself
	my $list = $checkParams->{list};
	if (ref($list) =~ /HASH/) {						# users declared as hash (user => {data}, ...)
		$checkMethod = sub {
			my ($user, $pass) = @_;
			$pass = &$Silvestris::User::CryptAlgo($pass);
			if (! $list->{$user}) { return (401, "Unknown user : '$user'"); } 
			elsif ($list->{$user}{'pass'} ne $pass) { return (401, "Wrong  password for user '$user'"); } 
			else { return (0, undef, $list->{$user}{'group'}); }
		};
	} elsif (ref($list) =~ /ARRAY/) {				# users declared as array ({user: user1, data...}, {user: user2, data, ...}, ...)
		$checkMethod = sub {
			my ($user, $pass) = @_;
			$pass = &$Silvestris::User::CryptAlgo($pass);
			my $data = undef; foreach (@$list) { $data = $_ if $_->{user} eq 'user'; }
			if (! $data) { return (401, "Unknown user : '$user'"); } 
			elsif ($data->{'pass'} ne $pass) { return (401, "Wrong  password for user '$user'"); } 
			else { return (0, undef, $data->{'group'}); }
		};
	} else {
		die "Incorrect list of users for authentication.";
	}
} elsif ($checkParams->{type} =~ /rules/i) {		# a set of rules defines if user exists or not
	my $userRule = $checkParams->{rules}{user} || '^u(ser)?(\d+)';	# default rule : user is u1,u2,... or user1,user2, ...
	my $passRule = $checkParams->{rules}{pass} || '^p(ass)?(w(or)d)?$id';	# default rule : password is pN, passN, pwdN or passwordN with N = user number
	my $groupRule = $checkParams->{rules}{group} || 'group$id';
	$checkMethod = sub {
		my ($user, $pass) = @_;
		my $id = $2 if $user =~ /$userRule/o;	
		my $requiredPass = eval qq("$passRule") if $id; # convert $id if needed
		if (!$id) { return (401, "Unknown user : '$user'"); } 
		elsif ($pass !~ /$requiredPass/) { return (401, "Wrong  password for user '$user'"); } 	
		else { return (0, undef, (eval qq("$groupRule"))); }
	};
} elsif ($checkParams->{type}) {	# unknown
	die "Unknown authentication method : $checkParams->{type}.";
}

$Silvestris::User::AuthMethod = $checkMethod; # used by test-auth application

# ----------- Now the hook 

hook before => sub {
  my ($user, $pass, $bearer);
	# 1. Identification: check if user has provided login and password
	if (my $auth = request->header('Authorization')) { # WWW-Basic authentication
		my ($mode, $base64) = split(/\s/, $auth);
		if ($mode =~ /Basic/i) {
			($user, $pass) = split(/:/, MIME::Base64::decode_base64($base64));
		} elsif ($mode =~ /Bearer/i) {
			($user, $pass) = split(/:/, $Silvestris::Users::Bearers{$base64});
			$bearer = $base64;
		}
	} elsif (ref($authConf->{'id-methods'}) and grep { /url|param/i } @{$authConf->{'id-methods'}}) {
		# User and password in the URL or HTTP-POST content. Support of this must be explicitly declared in config.yml
		$user = param('user'); $pass = param('password');
	}
	# 2. Authentication : are info provided by the user correct?
	if ($user and $pass) {
		my ($status, $message, $group);
		if (defined $Silvestris::Users::Cache{$user}) {
			if ($Silvestris::Users::Cache{$user}[0] ne $pass) {
				# Use check method, because passord may have been changed in the db
				($status, $message, $group) = &$checkMethod ($user, $pass);
				unless ($status) { $Silvestris::Users::Cache{$user}[0] = $pass; } # Success : change password in memory
				else { ($status, $message, $group) = (401, "Wrong password for user $user", undef); } # Failed : password is still not correct
			} else {	# Password is as in memory: no need to call the database 
				($status, $message, $group) = (0, undef, $Silvestris::Users::Cache{$user}[1]);       
				$Silvestris::Users::Cache{$user}[2] = time();
			}
		} else {
			($status, $message, $group) = &$checkMethod ($user, $pass);
			$Silvestris::Users::Cache{$user} = [ $pass, $group, time() ] unless $status > 400 or $checkParams->{type} =~ /file|config|rules/i; # keep in cache to avoid calling checkMethod
		}
		# Error cases
		if ($status > 400) { my $fmt = Silvestris::Cyclotis::Format->for_dancer_params(); status $status; halt $fmt->reencode (1, $fmt->produce_error($message)); }
		# Success : the user is correct, but now the application should decide if he is allowed to access the resource or not.
		var auth_user => $user if $user; var auth_group => $group if $group;
		if (defined($authConf->{bearer}) and ("$authConf->{bearer}{use}" !~ /no/) and not $bearer) {    # create a bearer 
			unless (("$authConf->{bearer}{use}" =~ /nego/) and not defined request->header('X-Support-Bearer')) {
				my @chars = ('0'..'9', 'A'..'F');
				my $len = $authConf->{bearer}{length}; my $string;
				$string .= $chars[rand @chars] while ($len--);
				$Silvestris::Users::Bearers{$string} = "$user:$pass:" + time();
				header 'X-Bearer' => $string;
			}
		}
	} else {
		# User is not authenticated. But is it really needed?
		my %require = (any => ':none'); %require = %{$authConf->{'permissions'}} if defined $authConf->{'permissions'};
		my @T = keys %require; while (my $k = shift(@T)) {
			if ($k =~ /^(contents|meta|admin)$/) {  # convert to flat
				while (my ($k1,$v) = each (%{$require{$k}})) {
					$require{"$k.$k1"} = $v;
				}
				delete $require{$k};
			}
		}
		my $requireAny = $require{any}; delete $require{any}; # treated after the loop
		if (ref($requireAny) and $requireAny->{ip}) {  # special case: produce halt, not a request for authentication!
			my $remote = request->address(); my $valid = undef;
			if (ref($requireAny->{ip}) =~ /ARRAY/) { $valid = scalar grep { $remote eq $_ } @{$requireAny->{ip}}; }
			else { $valid = ($remote eq $requireAny->{ip}); }
			unless ($valid) { status 401; my $fmt = Silvestris::Cyclotis::Format->for_dancer_params(); halt $fmt->reencode (1, "Forbidden for this host ($remote)"); }
			else { delete $requireAny->{ip}; $requireAny = ':none' unless scalar %{$requireAny}; }
		}
		my $query = request->uri; $query =~ s/\?.+$/\?/;
		while (my ($type, $need) = each (%require)) {
			my $urlPattern = '.'; if ($type ne 'any') {
				my ($cat, $op) = split(/\./, $type); 
				if ($op eq 'any') {
					my %all = (); foreach my $v (values(%{config->{url}{schemas}{$cat}})) {
						if (ref($v)) { foreach my $v1 (@$v) { $all{$v1}++; } } else { $all{$v}++; }
					}
					$urlPattern = join('|', map { "($_)" } keys(%all));
				} 
				else { $urlPattern = config->{url}{schemas}{$cat}{$op}; }
				$urlPattern =~ s/(:\w+)/"(.+?)"/ge; $urlPattern .= '(\?|$)';
			}
			if ($query =~ /^$urlPattern/) {
				return if lc($need) eq ':none';
				sub containsUser {
					my $need0 = shift;
					return 0 unless ref $need0;
					if (ref($need0) =~ /HASH/) { return (defined $need0->{user}) || (defined $need0->{group}); } # true if defined
					if (ref($need0) =~ /ARRAY/) { 
						foreach my $need1 (@$need0) { my $t = containsUser($need1); return $t if $t; }
						return 0; # none found
					}
					return 1; # scalar ref: no meaning for the moment
				}
				return unless containsUser($need); 
			} # if
		} # while
		return unless $requireAny ne ':none'; 
		# If we did not find query type or if need was not set to :none, we must ask for authentication
		status 401;
		header ( 'WWW-Authenticate' => sprintf('Basic realm="%s"', ($authConf->{realm} || 'Cyclotis Authentication')) );
		my $fmt = Silvestris::Cyclotis::Format->for_dancer_params(); 
		halt $fmt->reencode (1, $fmt->produce_error('Authentication required'));
	}
};

dance();

=head1 LICENSE

Copyright 2014-2018 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
