=encoding utf-8
=head1 DESCRIPTION

This is a SOAP service to access Cyclotis database. Runnable via SOAP::Lite deamon or any compatible web server, such as Apache::SOAP

=head1 USAGE (Server-side)

=head2 Via SOAP::Lite

  use SOAP::Transport::HTTP;

  SOAP::Transport::HTTP::CGI   
    -> dispatch_to('Silvestris::Cyclotis::SoapServer')     
    -> handle;

=head2 Apache configuration

  <Location /cyclotis>
    SetHandler perl-script
    PerlHandler Apache::SOAP
    PerlSetVar dispatch_to "/where/is/your/server, Silvestris::Cyclotis::SoapServer"
    PerlSetVar options "compress_threshold => 10000"
  </Location>

=cut
package Silvestris::Cyclotis::SoapServer;

use SOAP::Lite;
@ISA = qw(Exporter SOAP::Server::Parameters);

BEGIN { require 'soap-config.pl'; require Time::HiRes if $Silvestris::Cyclotis::SoapServer::CONFIG{debug}{time}; }
use DBI;

our %CMD = (
	# Text search modes 
	fuzzy => 'src % :txt', exact => 'src = :txt', like => 'src like :txt', 
    # Date search modes
    all => 'date >= :txt',
	# Stemmed string searches (concordance)
	concSrc => "to_tsvector(:lexer,src) @@ plainto_tsquery(:lexer,:txt)", 
	concTra => "to_tsvector(:lexer,tra) @@ plainto_tsquery(:lexer,:txt)",
	glosSrc => "to_tsvector(:lexer,src) @@ to_tsquery(:lexer,:txt)"	
);
# ISO-639-1 language codes to Postgresql's tsvector english names
our %HLANGS = ( en => 'english', es => 'spanish', fr => 'french', de => 'german', da => 'danish', nl => 'dutch', fi => 'finnish',
                hu => 'hungarian', it => 'italian', nb => 'norwegian', nn => 'norwegian', no => 'norwegian', pt => 'portugese', 
                ro => 'romanian', ru => 'russian', sv => 'swedish', tr => 'turkish' );

=head1 USAGE (Client-side)

=head2 Sample search query client

   use SOAP::Lite;
   use Data::Dumper;
   
   $res = SOAP::Lite -> uri ("urn:Silvestris/Cyclotis/SoapServer")
       -> proxy ("http://localhost/cyclotis")
	   -> find (database => "Cyclotis", table => "mem", text => "test", command => "fuzzy");
   print Dumper ($res->result || $res->fault);
  
=cut
sub find {
	my $self = shift; 
	my $env = pop; my %h = @_;
	my $db = $env->valueof('//database') || $h{database} || @_[0];
	my $table = $env->valueof('//table') || $h{table} || @_[1];
	my $query = $env->valueof('//text') || $h{text} || @_[2];
	my $cmd = $env->valueof('//command') || $h{command} || @_[3];

    if ($cmd =~ /^glos/) { # Glossary search : use operator OR instead of AND
       my @TAB = ($query =~ /\p{Letter}+/g); 
       my %TAB = (); $TAB{lc($_)}++ foreach (@TAB); @TAB = keys(%TAB);     # Removes dupplicates
       $query = join(' | ', @TAB);
    }	
	$cmd = $CMD{$cmd} || die "Unknown command: $cmd";
	my $dbh = DBI->connect_cached(@{$Silvestris::Cyclotis::SoapServer::CONFIG{databases}{$db}}) 
		or die "Cannot connect to $db : $DBI::errstr";
	my $sql = "select *, similarity(src,:txt) as score from $table where $cmd";
	$sql =~ s/src/tra/ if $cmd =~ /tra/;
	my $startTime = Time::HiRes::time() if $Silvestris::Cyclotis::SoapServer::CONFIG{debug}{time};
	my $st = $dbh->prepare_cached ($sql) or die "prepare: $DBI::errstr";
		
	if ($sql =~ /:lexer/) { # May use language-specific parsing
		my $lang = undef; $lang = param("lang$1") if $cmd =~ /(Src|Tra)$/; $lang ||= param('lang');
		$lang = lc (substr($lang, 0, 2));
		$st->bind_param (':lexer', $HLANGS{$lang} || 'simple');
	}
	$sql =~ s/similarity\(.*\)/1/ if $cmd =~ /all/;
	&{$Silvestris::Cyclotis::SoapServer::CONFIG{debug}{sql}} ($sql) if $Silvestris::Cyclotis::SoapServer::CONFIG{debug}{sql};
	$st->bind_param (':txt' => $query) or return $DBI::errstr;
	$st->execute () or die "execute: $DBI::errstr";
	
	my @RES;
	while (my $line = $st->fetchrow_hashref) {
		push (@RES, $line);
	}
	&{$Silvestris::Cyclotis::SoapServer::CONFIG{debug}{time}} (Time::HiRes::time() - $time) if $Silvestris::Cyclotis::SoapServer::CONFIG{debug}{time};
    return { Cyclotis => { version => 1 }, result => \@RES };
}

=head2 Sample save query client

   use SOAP::Lite;
   use Data::Dumper;
   
   $res = SOAP::Lite -> uri ("urn:Silvestris/Cyclotis/SoapServer")
       -> proxy ("http://localhost/cyclotis")
	   -> save (database => "Cyclotis", table => "mem", src => "Essai", tra => "try", author => "me"); 
   print Dumper ($res->result || $res->fault);
  
=cut
sub save {
	my $self = shift; 
        my $env = pop; my %h = @_;
	my $db = $env->valueof('//database') || $h{database} || @_[0];
	my $table = $env->valueof('//table') || $h{table} || @_[1];
	my $src = $env->valueof('//src') || $h{src} || @_[2];
	my $tra = $env->valueof('//tra') || $h{tra} || @_[3];
	my $author = $env->valueof('//author') || $h{author} || @_[4];
	my $context = $env->valueof('//context') || $h{context};

	my $dbh = DBI->connect_cached(@{$Silvestris::Cyclotis::SoapServer::CONFIG{databases}{$db}}) 
		or die "Cannot connect to $db : $DBI::errstr";
	my $sql = $context ? "insert into $table values (?,?,?,?,NOW())" : "insert into $table values (?,?,?,NOW())";
	my $st = $dbh->prepare_cached ($sql) or die "prepare: $DBI::errstr";
	my @params = ($src,$tra,$author); push(@params, $context) if defined $context;
		
	my $startTime = Time::HiRes::time() if $Silvestris::Cyclotis::SoapServer::CONFIG{debug}{time};
	my $res = $st->execute (@params) or die "execute: $DBI::errstr";
	&{$Silvestris::Cyclotis::SoapServer::CONFIG{debug}{time}} (Time::HiRes::time() - $time) if $Silvestris::Cyclotis::SoapServer::CONFIG{debug}{time};
	
    return { Cyclotis => { version => 1 }, result => $res };
}

=head2 Sample delete query client

	    use SOAP::Lite;
		use Data::Dumper;
		
		$res = SOAP::Lite -> uri ("urn:Silvestris/Cyclotis/SoapServer") 
			-> proxy ("http://localhost/cyclotis")
                 -> delete (database => "Cyclotis", table => "mem", src => "Essai", context => 'xxx'); 
		print Dumper ($res->result || $res->fault);
  
=cut
sub delete {
	my $self = shift; 
        my $env = pop; my %h = @_;
	my $db = $env->valueof('//database') || $h{database} || @_[0];
	my $table = $env->valueof('//table') || $h{table} || @_[1];
	my $src = $env->valueof('//src') || $h{src} || @_[2];
	my $context = $env->valueof('//context') || $h{context};

	my $dbh = DBI->connect_cached(@{$Silvestris::Cyclotis::SoapServer::CONFIG{databases}{$db}}) 
		or die "Cannot connect to $db : $DBI::errstr";
	my $sql = $context ? "delete from $table where src = ? and context = ?" : "delete from $table where src = ?";
	my $st = $dbh->prepare_cached ($sql) or die "prepare: $DBI::errstr";
	my @params = ($src); push(@params, $context) if defined $context;
		
	my $startTime = Time::HiRes::time() if $Silvestris::Cyclotis::SoapServer::CONFIG{debug}{time};
	my $res = $st->execute (@params) or die "execute: $DBI::errstr";
	&{$Silvestris::Cyclotis::SoapServer::CONFIG{debug}{time}} (Time::HiRes::time() - $time) if $Silvestris::Cyclotis::SoapServer::CONFIG{debug}{time};
	
    return { Cyclotis => { version => 1 }, result => $res };
}

1;

=head1 LICENSE

Copyright 2013 Silvestris Project (L<http://www.silvestris-lab.org/>)

Licensed under the EUPL, Version 1.1 or  as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut

