=head2 Format 'Envelope'

Overrides an xml file format to enable sending error messages

=cut
package Silvestris::Cyclotis::Format::nodep::Envelope;
use parent Silvestris::Cyclotis::Format::nodep;

=head3 Available parameters

=over 1

=item subformat

Reference to an already defined format.

=item onTrue

Boolean: indicates whenever we want envelope on correct response, or not.

=item envelope-namespace

If true, all elements from the envelope will be prefixed (prefix is env:, unless parameter is explicit name) 

=back

=cut
sub new {
    my (undef, %params) = @_;
    return bless \%params;
}

sub MIME { 'application/xml' }

sub prefix_envelope {
	my $self = shift; my $res = shift;
	if ($self->{'envelope-prefix'}) {
		my $prefix = $self->{'envelope-prefix'}; $prefix = $self->{'envelope-prefix'} = 'env' if $prefix and $prefix !~ /^[a-z][\w\-]+$/i;
		$res =~ s!<result(\s|>)!<result xmlns:$prefix="http://www.silvestris-lab.org/cyclotis"$1!;
		foreach my $name ('result', 'timestamp', 'formats', 'url-list', 'url', 'tables', 'contents', 'query', 'matches-count') {	# change only markups from envelope
			$res =~ s!<(/?)$name(\s|>)!<$1$prefix:$name$2!gs;
		}
	}
	elsif ($self->{'envelope-namespace'}) {
		$res =~ s!<result(\s|>)!<result xmlns="http://www.silvestris-lab.org/cyclotis"$1!;	
	}
 return $res;
}

sub produce_error {  
	return $_[0]->prefix_envelope(<< "EOF");
<?xml version='1.0' encoding='utf-8'?>
<response status="ERROR">$_[1]</response>
EOF
}

sub namespace_uri { 'http://www.silvestris-lab.org/cyclotis' }

sub header {
  my ($self, @param) = @_;
  my $res = $self->{subFormat}->header(@param); return $res unless $self->{onTrue};
  $res =~ s!^<\?xml.+?\?>\n!!;
  $res =~ s!<\!.+?>\n!!;
  if ($self->{'contents-prefix'}) {
		my $prefix = $self->{'contents-prefix'}; $prefix = $self->{'contents-prefix'} = substr(ref($self->{subFormat}), rindex($self->{subFormat},'::') + 2) if $prefix and $prefix !~ /^[a-z][\w\-]+$/i;
		my $URI = $self->{subFormat}->namespace_uri;
		$res =~ s!<([a-z][\w\-]+)([\s>])!<$1 xmlns:$prefix="$URI" $2!;	# adds namespace specification only to first markup
		$res =~ s!<(/?)([\w\-]+)(\s|>)!<$1$prefix:$2$3!gs;	# adds prefix to all markups, including first one
  }
  elsif ($self->{'contents-namespace'}) {
		my $URI = $self->{subFormat}->namespace_uri;
		$res =~ s!<([\w\-]+)([\s>])!<$1 xmlns="$URI" $2!;	# adds namespace specification only to first markup
  }
  my $name = substr(ref($self->{subFormat}), rindex($self->{subFormat},'::') + 2);
  return $_[0]->prefix_envelope(<< "EOF");
<?xml version='1.0' encoding='utf-8'?>
<response status="OK" type="CONTENTS">
<query>$param[0]</query>
<contents format="$name">
$res
EOF
}

sub produce_line {
  my ($self, @param) = @_;
	my $res = $self->{subFormat}->produce_line(@param); return $res unless $self->{onTrue};
	my $prefix = $self->{'contents-prefix'}; $res =~ s!<(/?)([\w\-]+)(\s|>)!<$1$prefix:$2$3!gs if $prefix;	# adds prefix to all markups
	return $res;
}

sub footer {
  my ($self, @param) = @_;
	my $res = $self->{subFormat}->footer(@param); return $res unless $self->{onTrue};
	my $prefix = $self->{'contents-prefix'}; $res =~ s!<(/?)([\w\-]+)(\s|>)!<$1$prefix:$2$3!gs if $prefix;	# adds prefix to all markups
	return "$res\n" . $self->prefix_envelope("</contents><matches-count>$param[0]</matches-count></result>\n");
}

sub status_line { 
  my ($self, @param) = @_;
	my $res = $self->{subFormat}->status_line(@param); return $res unless $self->{onTrue};
	my $prefix = $self->{'contents-prefix'}; $res =~ s!<(/?)([\w\-]+)(\s|>)!<$1$prefix:$2$3!gs if $prefix;	# adds prefix to all markups
	return $res;
}

sub read_line {
  my ($self, @param) = @_;
	return $self->{subFormat}->read_line(@param);
}

# ---- responses which could not be produced by the subFormat

sub produce_info { 
	my ($self, $formats, $url, $allow, $timestamp, $tables) = @_;
  my $allowStruct = $allow->{'table-struct-display'} || '*.*';
  my $allowFindPerPost = $allow->{'find-per-post'} ? 'true' : 'false';
  my $allowSavePerGet = $allow->{'save-per-get'} ? 'true' : 'false';
  $tables = "<tables read='$1' write='$2' />" if $tables =~ /^(.+),(.+)$/;
	return $self->prefix_envelope(<< "EOF");
<response type="INFO" status="OK">
	<timestamp>$timestamp</timestamp>
	<formats default="$formats->{default}" />
	<url>
		<struct allowed="$allowStruct">$url->{struct}</struct>
	 	<find can-post="$allowFindPerPost">$url->{contents}{find}</find>
		<save can-get="$allowSavePerGet">$url->{contents}{save}</save>
		<delete>$url->{contents}{delete}</delete>
	</url>
	$tables
</response>
EOF
}

sub struct_info {
	my ($self, %tables) = @_; my $TEXT = "";
	while (my ($tableName, $ref) = each (%tables)) {
		my %options = %$ref; my $cols = $options{fields}; my $props = $options{propsstore}; delete @options{'fields','table_schema','table_name', 'propsstore'};			
		$TEXT .= "\t<table name='$tableName'>\n\t\t<columns>\n";
		while (my ($colName, $colType) = each (%$cols)) {
			if (ref($colType) =~ /Composite/) {
				$TEXT .= "\t\t\t<col name='$colName' type='$colType->{name}' struct='composite'>\n";
				$TEXT .= "\t\t\t\t<field name='$_' />\n" foreach @{$colType->{fields}};
				$TEXT .= "\t\t\t</col>\n"
			} else {
				$TEXT .= "\t\t\t<col name='$colName' type='$colType' />\n";
			}
		}
		$TEXT .= "\t\t</colums>\n\t\t<options>\n";
		while (my ($optName, $optVal) = each (%options)) {
      if (ref ($optVal) =~ /HASH/) {
  			$TEXT .= "\t\t\t<option name='$optName'>\n";
        while (my ($k,$v) = each(%$optVal)) { $TEXT .= "\t\t\t\t<$k>$v</$k>\n"; }
  			$TEXT .= "\t\t\t</option>\n";
      } else {
  			$TEXT .= "\t\t\t<option name='$optName' value='$optVal' />\n";
      }
		}
		$TEXT .= "\t\t</options>\n\t\t<properties>\n";
		while (my ($propName, $propType) = each (%$props)) {
			$TEXT .= "\t\t\t<property name='$propName' value='$propType' />\n";
		}
		$TEXT .= "\t\t</properties>\n";
		$TEXT .= "\t</table>";
	}
	return $self->prefix_envelope(<< "EOF");
<response type="TABLE-STRUCTURE" status="OK">
$TEXT
</response>
EOF
}

sub show_status {
	my ($self, $level, $types, $Queries, $Lib) = @_; my $res = "<response type='SERVER-STATUS' status='OK'>\n";
	unless ($types !~ /cache/) {
		$res .= "\t<tables count='" . scalar (keys (%Silvestris::Cyclotis::Database::Table::CACHE)) . "'";
		if ($level <= 1) { $res .= " />\n"; } else {
			$res .= ">\n";
			$res .= "\t\t<table name='$_' />\n" foreach keys (%Silvestris::Cyclotis::Database::Table::CACHE);
			$res .= "\t</tables>\n";
		}		
		$res .= "<queries count='" . scalar(@$Queries) . "'";
		if ($level <= 1) { $res .= " />\n"; } else {
			$res .= ">\n";
			foreach (@$Queries) {
				if (/^(insert|update|select|delete)\s+([\w\.]+)/) {
					$res .= "\t\t<query type='$1' table='$2' />\n";
				} else {
					$res .= "\t\t<query code='$_' />\n";
				}
			}
			$res .= "\t</queries>\n";
		}
	}
	unless ($types !~ /libs/) {
		$res .= "\t<libraries>\n";
		while (my ($lib, $hash) = each (%$Lib)) {
			$res .= "\t\t<library name='$lib' location='$hash->{location}' version='$hash->{version}' date='$hash->{date}' />\n";
		}
		$res .= "\t</libraries>\n";
	}
	return $self->prefix_envelope ($res . "\n</response>");	
}


1;

=head1 LICENSE

Copyright 2015-2016 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or  as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
