﻿=head1 DESCRIPTION

These are the routes to access meta-information, i.e. info about the tables rather than contents.

=cut

use Dancer; 

use Silvestris::Cyclotis::Format;
use Silvestris::Dancer::MyRoutes;

=head2 GET /meta/(:db/:table/)?info(.:fmt)?

Get info about the server: URL mappings, authorizations, defaults.

=head3 Using inheritance

The user may want to do queries in a parent table while writing in the specific one.
To do this you can either

=over 1

=item - add a parameter C<read> containing table name
=item - specify read and write tables as C<:table> parameter with C<,> as separator
=item - specify a number of parents, then C<^> and the name of write table. This works only if you make use of single inheritance.
=item - specify a number of parents using parameter "inherit"

=back

=head3 ADDITIONNAL PARAMETERS

findfmt, savefmt or structfmt: Will replace :fmt by the given value when generating respectively find, save and struct url.

=cut
sv_route 'meta.info' => get => '/meta/((:db/)?:table/)?info(.:fmt)?' => sub {
	my $fmt = Silvestris::Cyclotis::Format->for_dancer_params(); content_type $fmt->MIME() . "; charset=" . ($fmt->{encoding} || 'utf-8'); 
	my $schema = param('schema'); my $table = param('table'); 
	my ($timestamp) = database->selectrow_array('SELECT extract(epoch from now())');
	my %urlCopy = %{config->{url}{schemas}}; %urlCopy = map { $_ => ref($urlCopy{$_}) ? { %{$urlCopy{$_}} } : $urlCopy{$_} } keys(%urlCopy); # deep copy
	if ($schema or $table or param('read')) {
		if (my $check = config->{'meta-info'}{check}) {	# check that the table exists, else produce error
			unless ($Silvestris::Cyclotis::Database::Table::CACHE{param('db') . '.' . $schema . '.' . $table}) {
				my ($Cschema,$Ctable) = ($1,$2) if $table =~ /^(?:.+[,\^\<])?(?:(\w+)\.)?(\w+)$/; $Cschema ||= 'public';
				$check = 'select * from ' . config->{'meta-info'}{view} . ' where table_schema=? and table_name=?' if $check eq 'meta';
				$check = 'select * from pg_catalog.pg_tables where schemaname=? and tablename=?' if $check eq 'pg';
				$check = 'select * from information_schema.tables where table_schema=? and table_name=?' if $check eq 'information_schema';
				$check = "select * from $schema.$table where src % ? or tra % ? or 1 = 1" if $check eq 'data';
				if (config->{debug}{sql}) { my $debug = sv_debug('sql'); &$debug("SQL = $check ($schema,$table)"); }
				my $checkReq = database(param('db'))->prepare_cached($check) or return $fmt->produce_error($DBI::errstr);
				$checkReq->execute(lc($Cschema), lc($Ctable)) or return $fmt->produce_error($DBI::errstr);
				my @t = $checkReq->fetchrow_array(); return $fmt->produce_error($DBI::errstr || "Memory $Cschema.$Ctable does not exist") unless @t; # only check for existence
			}
		}
		my $read = param('read'); my $fallBack = param('fallback'); 
		unless ($read) {
			$read = $1 if $table =~ /^([\w\.]+)[,<]/; my $inherit = param('inherit');
			if ($table =~ /^(\d+)\^(\^?)([\w\.]+)$/) { $inherit = $1; $fallBack ||= $2; $read = $table = $3; } else { $read ||= $table; }
			if (defined $inherit) {
				my $inhReq = database->prepare_cached("
	    SELECT pg_namespace.nspname || '.' || pg_class.relname FROM pg_catalog.pg_inherits
         INNER JOIN pg_catalog.pg_class ON (pg_inherits.inhparent = pg_class.oid)
         INNER JOIN pg_catalog.pg_namespace ON (pg_class.relnamespace = pg_namespace.oid)
         WHERE inhrelid = ?::regclass
				");
				for (my $cpt = $inherit; $cpt > 0; $cpt--) {
					$inhReq->execute($read);
					($read) = $inhReq->fetchrow_array;
					1 while $inhReq->fetchrow_array;
					$inhReq->finish;
				}
			}
		}
		if ($read) {	# we have a view to read and a table to write
			sub correctUrl {
				my ($url, $name, $defaultSchema) = @_;
				my ($s, $t) = split(/\./,$name); unless ($t) { $s = $defaultSchema || 'public'; $t = $name; }
				if ($$url =~ s!:schema!$s!g) { $$url =~ s!:table!$t!g; } else { $$url =~ s!:table!$name!g; }
			}
			# find  and info are related to read table except for fallback
			if ($table =~ /</) { correctUrl(\$urlCopy{contents}{find}, $table, $schema); correctUrl(\$urlCopy{meta}{info}, $table, $schema); }
			elsif ($fallBack) { correctUrl(\$urlCopy{contents}{find}, "$read<$table", $schema); correctUrl(\$urlCopy{meta}{info}, "$read<$table", $schema); }
			else { correctUrl(\$urlCopy{contents}{find}, $read, $schema); correctUrl(\$urlCopy{meta}{info}, $read, $schema); }
			correctUrl(\$urlCopy{meta}{struct}, $read, $schema);  # struct is always on read table
			# save and delete are related to write table
			my $write = param('write') || $table; $write =~ s/^.+?[,<]//;
			correctUrl(\$urlCopy{contents}{save}, $write, $schema);
			correctUrl(\$urlCopy{contents}{delete}, $write, $schema);
		} elsif ($table) {	# we want to read/write in the same table
			if (grep { /:schema/ } values %{$urlCopy{contents}}) {
				unless ($schema) { ($schema,$table) = ($1,$2) if $table =~ /^(\w+)\.(\w+)$/; $schema ||= 'public';	}
				s!:schema!$schema!g foreach (values (%{$urlCopy{contents}}), values (%{$urlCopy{meta}})); 
			}
			s!:table!$table!g foreach (values (%{$urlCopy{contents}}), values (%{$urlCopy{meta}})); 
		}
		if (my $db = param('db')) { s!:db!$db!g foreach (values (%{$urlCopy{contents}}), values (%{$urlCopy{meta}})); }
		else { foreach (values (%{$urlCopy{contents}}), values (%{$urlCopy{meta}})) { s!:db!!g; } }
		foreach (values (%{$urlCopy{contents}}), values (%{$urlCopy{meta}})) { s!\(([^:]+?)\)\?!$1!g; s!/+!/!g; }
	}
	if (my $findFmt = param('findfmt')) { $urlCopy{contents}{find} =~ s/:fmt/$findFmt/; }
	if (my $saveFmt = param('savefmt')) { $urlCopy{contents}{save} =~ s/:fmt/$saveFmt/; }
	if (my $structFmt = param('structfmt')) { $urlCopy{meta}{struct} =~ s/:fmt/$structFmt/; }
	my $addPrefix = param('add_prefix'); my $delPrefix = param('del_prefix'); $delPrefix = [$delPrefix] if $delPrefix and not ref($delPrefix);
	if (param('mapping')) {
		my $mapping = config->{url}{mappings}{param('mapping')} or return $fmt->produce_error("Unknown mapping: " . param('mapping'));
		$addPrefix ||= $mapping->{add_prefix};
		if ($mapping->{del_prefix}) {
			$delPrefix ||= []; push (@$delPrefix, ref($mapping->{delPrefix}) ? @{$mapping->{del_prefix}} : $mapping->{del_prefix});
		}
	} else {
		foreach my $mapping (values (%{config->{url}{mappings}})) {
			my $detect = $mapping->{'detect-http-header'} or next;
			while (my ($k, $v) = each (%$detect))  {
				if (request->header($k) eq $v) {
					$addPrefix ||= $mapping->{add_prefix};
					if ($mapping->{del_prefix}) {
						$delPrefix ||= []; push (@$delPrefix, ref($mapping->{delPrefix}) ? @{$mapping->{del_prefix}} : $mapping->{del_prefix});
					}
				}
			}
		}
	}
	if ($addPrefix) { $_ = $addPrefix . $_ foreach (values (%{$urlCopy{contents}}), @urlCopy{'struct','info'}); }
	if ($delPrefix) {
		my @delPrefixes; if (ref($delPrefix)) { @delPrefixes = @$delPrefix; } else { @delPrefixes = ($delPrefix); }
		foreach my $prefix0 (@delPrefixes) {
			foreach (values (%{$urlCopy{contents}}), @urlCopy{'struct','info'}) {
				$_ = substr($_, length($prefix0)) if substr($_, 0, length($prefix0)) eq $prefix0;
			}
		}
	}
	my $formats = config->{formats}; $formats ||= {}; $formats->{default} ||= 'line';
	return $fmt->produce_info($formats, \%urlCopy, config->{allow}, $timestamp, $table);
};

=head2 GET /

Default route is a shortcut for /meta/info without parameters

=cut
get '/' => sub {
	my $fmt = Silvestris::Cyclotis::Format->for_dancer_params(); content_type $fmt->MIME() . "; charset=" . ($fmt->{encoding} || 'utf-8'); 
	my $formats = config->{formats}; $formats ||= {}; $formats->{default} ||= 'line';
	return $fmt->produce_info($formats, config->{url}{schemas}, config->{allow});
};

set 'special-tables' => [ config->{'meta-info'}{view} || 'public.meta_info' ] unless config->{'special-tables'};

=head2 GET /struct/:table?.:fmt?

Get info about database structure : list tables

=cut
sv_route 'meta.struct' => get => '/meta/((:db/)?:table/)?struct(.:fmt)?' => sub {
	my $fmt = Silvestris::Cyclotis::Format->for_dancer_params(); content_type $fmt->MIME() . "; charset=" . ($fmt->{encoding} || 'utf-8'); 
	my $schema = param('schema'); my $table = param('table') || '';
	($schema,$table) = ($1, $2) if $table =~ /^(\w+)\.(\w+)$/; $schema ||= 'public';
	if ($table) { 
		return $fmt->produce_error ("Structure of this table cannot be displayed") 
			if grep { $_ eq $table } @{config->{'special-tables'}};
		my $struct = find Silvestris::Cyclotis::Database::Table (param('db'),$table,$schema); 
		$struct = $struct->clean(config->{'meta-info'}{'col-exclude'})->force(config->{'meta-info'}{'col-force'}) if $struct;
		return $fmt->produce_error ("Table $table not found") unless $struct;
		return $fmt->struct_info($table => $struct); 
	} else {
		my %tables = find Silvestris::Cyclotis::Database::Table (param('db'),undef, $schema);
		return $fmt->produce_error ($@) if $@;
		delete @tables{@{config->{'special-tables'}}};
		%tables = map { $_ => $tables{$_}->clean(config->{'meta-info'}{'col-exclude'})->force(config->{'meta-info'}{'col-force'}) } keys(%tables);
		return $fmt->struct_info(%tables); 
	}
};


dance();


=head1 LICENSE

Copyright 2013-2018 Silvestris Project (L<http://www.silvestris-lab.org/>)

Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: L<http://ec.europa.eu/idabc/eupl>

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
