#! /usr/bin/env perl

=head1 DESCRIPTION

Contains the functions which do the variable replacements

=head1 FUNCTIONS

=cut

package vr;  # variable replacer

use Dancer ':syntax';


=head2 %hash = fillDatabaseParams()

Extracts database params from config, then replace variable names by values from user parameters

=cut
sub fillDatabaseParams {
	my %spec = %{config->{database}}; foreach my $val (values (%spec)) { 
		unless (ref ($val)) { $val = interpolation($val); }
		else {
			next if ref($val) !~ /ARRAY/;
			foreach my $cond (@$val) {
				next if ref ($cond) !~ /HASH/ or not $cond->{type};
				if ($cond->{type} eq 'default') { $val = interpolation($cond->{value}); last; }  

				my $expr = interpolation($cond->{expression}); 
				if (($cond->{type} eq 'regex') and ($expr =~ interpolation($cond->{expected}))) { $val = interpolation($cond->{value}); last; }  
				if (($cond->{type} eq 'eq') and ($expr eq interpolation($cond->{expected}))) { $val = interpolation($cond->{value}); last; }  
			}
		}
	}
	return %spec;
}

=head2 %hash = reverseDatabaseParams($memory)

Find user-friendly parameters which can be used to be put in an URL

=cut
sub reverseDatabaseParams {
  my $memory = shift; my @params;
	foreach my $key ('host','port','catalog','table','user','pass') { 
		my $val = config->{database}{$key} or next;
		unless (ref ($val)) {  
			while ($val =~ /\$(\w+)/g) { push (@params, $1 => $memory->{$1}); }
		} else {
			foreach my $cond (@$val) {
				while ($cond->{expression} =~ /\$(\w+)/g) { push (@params, $1 => $memory->{$1}); }
			}
		}
	}
	return @params;
}

=head2 $v = interpolation($expr)

Translate $expr by replacing $xxx to param(xxx),
${xxx:0,2} by param(xxx).substring(0,2)

=cut
sub interpolation {
	my $expr = shift; my $vars = shift || {};
	$expr =~ s/\$(\w+)/$vars->{$1} || param($1) || alias($1)/eg;
	$expr =~ s/\$\{(\w+)\}/$vars->{$1} || param($1) || alias($1)/eg;
	$expr =~ s/\$\{(\w+):(\d+),(\d+)\}/substr($vars->{$1} || param($1),$2,$3) || alias($1)/eg;
	$expr =~ s/\$\{(\w+):(\d+)\+\}/substr($vars->{$1} || param($1) || alias($1),$2)/eg;
	$expr =~ s[\$\{(\w+)!(.+)\}][
    my $val = $vars->{$1} || param($1) || alias($1); my $op = $2;
    $val = uc($val) if $op =~ /U/;
    $val = ucfirst($val) if $op =~ /u/;
    $val = lc($val) if $op =~ /L/;
    $val = lcfirst($val) if $op =~ /l/;
    $val =~ tr/_-/  / if $op =~ /S/; # display with spaces
    $val =~ tr/_ /--/ if $op =~ /D/; # display with dashes
    $val =~ tr/- /__/ if $op =~ /I/; # identifier
    $val
   ]egx;
	return $expr;
}

=head2 $v = alias($v)

If variable is not defined, check inside list of creation parameters

=cut
sub alias {
	my $name = shift;
ALTERNATIVE:
	foreach my $conf (@{config->{'creation-parameters'}}) {
		my $alt = $conf->{alternatives} or next;
		foreach my $alt (@$alt) {
			my $set = $alt->{set} or next;
			foreach my $var (@$set) {
				next unless $var->{'var-name'} eq $name;
				my $value = $var->{value};
				while ($value =~ m!\$\{?(\w+)!g) {
					next ALTERNATIVE unless param($1);
				}
				return interpolation($value);
			}
		}
	}
}

=head2 @tab = possibleValues($expr)

Generates an array where each variable present in $expr is replaced by all possible values

=cut
sub possibleValues {
	my $expr = shift;
	if (ref($expr)) {  # conditionnal
		my @all = ();
		foreach my $cond (@$expr) {
			push (@all, possibleValues($cond->{value}));
		}
		return @all;
	} elsif ($expr =~ /\$(\w+)/) {
		my $varName = $1; my @all = ();
		if (param($varName)) {
			my $copy = $expr; $copy =~ s/\$$varName/param($varName)/ge;
			my @list = possibleValues($copy); foreach my $item (@list) { $item->{$varName} = param($varName); }
			push (@all, @list);
		} else {
			foreach my $val (@{config->{'possible-values'}{$varName}}) { 
				my $copy = $expr; $copy =~ s/\$$varName/$val/ge;
				my @list = possibleValues($copy); foreach my $item (@list) { $item->{$varName} = $val; }
				push (@all, @list);
			}
		}
		return @all;
	} else {
		return ( {value => $expr } );
	}
}

use DBI;

=head2 @tab = memoriesList()

Create a list of objects, one for each existing memory. 

If filters are given in config, if the user gave a parameter value for them only the memories with given value are returned.
Else, all memories are given ordered by the filter values.

=cut
sub memoriesList {
	my %MEM = ();
	foreach my $host (possibleValues (config->{database}{host})) {
		foreach my $port (possibleValues (config->{database}{port})) {
			foreach my $db (possibleValues (config->{database}{catalog})) {
				my %current = (%$host, %$port, %$db, host => $host->{value}, port => $port->{value}, catalog => $db->{value}); delete $current{value};
				my $DBI = DBI->connect("dbi:Pg:host=$host->{value};port=$port->{value};dbname=$db->{value}", 
					config->{database}{user}, config->{database}{pass}, { PrintError => 0, %{ config->{database}{options} || {} } });
				next unless $DBI; # do not fail when connection does not work
				
				my $st = $DBI->prepare ("select * from " . (ref(config->{list}{sql}{view}) ? config->{list}{sql}{view}{name} : config->{list}{sql}{view})); $st->execute;
			SQL_VIEW:
				while (my $hash = $st->fetchrow_hashref) {
					my %copy = (%current, %$hash); my %doFun;
					foreach my $var (@{config->{list}{table}}) {
						$copy{$var->{'var-name'}} = interpolation($var->{value}, \%copy) if defined $var->{value};
						next SQL_VIEW if $var->{'valid-if'} and $var->{'valid-if'} =~ m!^/(.+)/$! and $copy{$var->{'var-name'}} !~ /$1/;
						$doFun{$var->{'var-name'}} = $var->{if} if $var->{function} or $var->{'group-function'};
					}
					my $DestTab = \%MEM;
					if (defined config->{'filter-parameters'} and scalar @{config->{'filter-parameters'}}) {
						for (my $i = 0; $i < @{config->{'filter-parameters'}}; $i++) {
							my $var = ${config->{'filter-parameters'}}[$i];
							if (param($var->{'var-name'})) {
								next SQL_VIEW if $copy{$var->{'var-name'}} and (param($var->{'var-name'}) ne $copy{$var->{'var-name'}});	# Apply filter, if exists
							} elsif (defined ($doFun{$var->{'var-name'}}) and ($doFun{$var->{'var-name'}} eq 'filter')) {
								$doFun{$var->{'var-name'}} = 0;
							}
							unless ($DestTab->{$copy{$var->{'var-name'}}}) {
								if ($i < @{config->{'filter-parameters'}} - 1) { 	
									$DestTab = $DestTab->{$copy{$var->{'var-name'}}} = {};	# Not last: create empty hash
								} else {
									$DestTab = $DestTab->{$copy{$var->{'var-name'}}} = [];  # Last : create empty table
								}
							} else {
								$DestTab = $DestTab->{$copy{$var->{'var-name'}}};	# Get already existing hash/table
							}
						}
					} else {
						$DestTab = $MEM{__DEFAULT__} ||= [];
					}
					calcGroupFields (\%copy, \%doFun, $DBI);
					push (@$DestTab, \%copy);	# unless we called next SQL_VIEW
				}
				$DBI->disconnect;
			}
		}
	}
	return %MEM;
}

my %existFields = ();

sub calcGroupFields {
	my $mem = shift; my $doAll = shift;
    my $DBI = shift; my $DBI_DEL; my @fields; my %pos;
    my ($schema, $tableName);
    if (ref (config->{list}{sql}{view})) { $schema = $mem->{config->{list}{sql}{view}{fields}{'table-schema'}};  $tableName = $mem->{config->{list}{sql}{view}{fields}{'table-name'}}; }
    $schema ||= $mem->{table_schema} || $mem->{schemaname} || 'public'; $tableName ||= $mem->{table_name} || $mem->{tablename};
FIELDS_LOOP:
	foreach my $field (@{config->{list}{table}}) {
		next unless $field->{'group-function'};
		my $name = $field->{'var-name'};
		next if $field->{'if'} eq 'filter' and (! $doAll->{$name});
		# Now we really want this field
		$DBI ||= $DBI_DEL = DBI->connect("dbi:Pg:host=$mem->{host};port=$mem->{port};dbname=$mem->{catalog}", config->{database}{user}, config->{database}{pass}); 
		my $expr = $field->{'group-function'};
		if (ref ($field->{'db-field'})) {
			$expr .= "(coalesce(" . join(',',@{$field->{'db-field'}}) . "))";
			if ($field->{'check-exists'}) {
				foreach my $f0 (@{$field->{'check-exists'}}) {
					unless (defined $existFields{"dbi:Pg:host=$mem->{host};port=$mem->{port};dbname=$mem->{catalog};table=$schema.$tableName"}{$f0}) {
						# not yet known, check the tableName
						my $exist = $DBI->selectrow_array("SELECT column_name FROM information_schema.columns 
						   WHERE table_name='$tableName' and column_name='$f0' and table_schema='$schema'");
						$existFields{"dbi:Pg:host=$mem->{host};port=$mem->{port};dbname=$mem->{catalog};table=$schema.$tableName"}{$f0} = $exist ? 1 : 0;
					}
					$expr =~ s/$f0,// if $existFields{"dbi:Pg:host=$mem->{host};port=$mem->{port};dbname=$mem->{catalog};table=$schema.$tableName"}{$f0} == 0;
				}
				$expr =~ s/coalesce\((\w+)\)/$1/;
			}
		} elsif ($field->{'db-field'}) {
			$expr .= "($field->{'db-field'})";
		} else {
			$expr .= '(*)';
		}
		if ($expr =~ /^join/) {
			$expr = substr($expr, index($expr,'(') + 1); $expr =~ s/\s*\)$//;
			my $req = $DBI->prepare("select distinct $expr from $schema.$tableName"); $req->execute;
			$mem->{error} = $DBI::errstr if $DBI::errstr;
			my @res = (); while (my @t = $req->fetchrow_array) { push (@res,$t[0]); }
			if ($field->{'max-length'}) {			
				$mem->{$name} = '';
			RES_LOOP:
				while (my $x = shift(@res)) {
					if (length($mem->{$name}) < $field->{'max-length'}) { $mem->{$name} .= "$x,"; }
					else { $mem->{$name} .= "..."; last RES_LOOP; }
				}
				$mem->{$name} =~ s/,$//;
			} else {
				$mem->{$name} = join(',',@res);
			}
			next FIELDS_LOOP; # This is not a group field
		}
		push(@fields, $expr); $pos{$name} = $#fields;
	}
	return unless @fields;
    my @res = $DBI->selectrow_array ("select " . join(',', @fields) . " from $schema.$tableName");
	$mem->{error} = $DBI::errstr if $DBI::errstr;
	while (my ($name,$id) = each (%pos)) { $mem->{$name} = $res[$id]; }
    $DBI_DEL->disconnect if $DBI_DEL;
}

sub dbiSpec {
	my %spec = &fillDatabaseParams();
	my $dbiSpec = sprintf('dbi:Pg:host=%s;port=%i;dbname=%s', @spec{'host','port','catalog'});
	return $dbiSpec;
}

sub databaseSpec {	# dbcreator's syntax
	my %spec = &fillDatabaseParams();
	my $res = $spec{catalog}; $res = "$spec{host}:$spec{port}/$res" if $spec{host} || $spec{port}; 
	$res = "$spec{user}:$spec{pass}\@$res" if $spec{user} || $spec{pass};
	return $res;
}

1;

=head1 License

Copyright 2014-2015 Silvestris Project (http://www.silvestris-lab.org/)

Licensed under the EUPL, Version 1.1 or – as soon they will be approved by the European Commission - subsequent versions of the EUPL (the "Licence");
You may not use this work except in compliance with the Licence.
You may obtain a copy of the Licence at: http://ec.europa.eu/idabc/eupl

Unless required by applicable law or agreed to in writing, software distributed under the Licence is distributed on an "AS IS" basis,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the Licence for the specific language governing permissions and limitations under the Licence. 

=cut
